package pacManReloaded.client;

import static pacManReloaded.common.Command.BEGIN_GAME;
import static pacManReloaded.common.Command.CHANGE_DIRECTION;
import static pacManReloaded.common.Command.GET_CLIENT_INFO;
import static pacManReloaded.common.Command.GET_HIGHSCORE;
import static pacManReloaded.common.Command.GET_LOBBY_INFO;
import static pacManReloaded.common.Command.GET_MANUAL;
import static pacManReloaded.common.Command.JOIN_LOBBY;
import static pacManReloaded.common.Command.MESSAGE;
import static pacManReloaded.common.Command.MY_NICKNAME;
import static pacManReloaded.common.Command.PING;
import static pacManReloaded.common.Command.SEND_TO_ADDRESS;

import java.awt.geom.Point2D;
import java.awt.geom.Point2D.Float;
import java.io.IOException;
import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import pacManReloaded.common.CommandFactory;
import pacManReloaded.common.messages.ChatMessage;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.messages.GameFinishedMessage;
import pacManReloaded.common.messages.IMessage;
import pacManReloaded.common.messages.MapMessage;
import pacManReloaded.common.model.Ball;
import pacManReloaded.common.model.Direction;
import pacManReloaded.common.model.Gate;
import pacManReloaded.common.model.Ghost;
import pacManReloaded.common.model.GhostColor;
import pacManReloaded.common.model.Map;
import pacManReloaded.common.model.PacMan;
import pacManReloaded.common.model.PowerBall;
import pacManReloaded.common.model.Wall;
import pacManReloaded.server.GameState;
import pacManReloaded.server.Player;

/**
 * Client class creates client and handles its actions.
 */
public class Client {

  private Socket socket;
  private final int pingInterval = 5 * 1000; // 5s
  String serverAddress;
  int serverPort;
  private boolean isConnected;
  private ClientGUI clientGUI;
  private ClientThread clientThread;
  private String tempNickname; // used until we can login to the server and get a real name
  private Map map;

  private GameState gameState;

  private HashMap<String, Player> actors;


  /**
   * Constructor of Client, opens a new socket to be able to connect to the server 3 args: - 1: ip
   * address - 2: port of the socket - 3: name (optional, if no third argument: username of computer
   * used) let you choose your name.
   *
   * @param nickname - name of your client
   * @param ipAddress - ip-address of the server
   * @param port - port of the server
   */
  public Client(String nickname, String ipAddress, int port) {
    this.tempNickname =
        nickname == null || nickname.trim().length() == 0 ? System.getProperty("user.name")
            : nickname;
    this.serverAddress = ipAddress;
    this.serverPort = port;
    clientGUI = ClientGUI.initGUI();
    clientGUI.setClient(this);
    actors = new HashMap<>();
  }

  /**
   * method which disconnects the client from the server, thread stops isConnected status updated
   */
  public void disconnectFromServer() {
    System.out.println("Disconnecting from server");
    if (clientThread != null) {
      clientThread.close();
      System.out.println("Thread stopped");
      isConnected = false;
    }
  }

  /**
   * method which sends the message in the chat
   *
   * @param message - message to be send
   */
  public void sendChatMessage(String message) {
    System.out.println("Sending chat: " + message);
    ArrayList<String> params = new ArrayList<>();
    params.add(clientThread.getMyName());
    params.add(message);
    IMessage gameMessage = CommandFactory.createMessage(MESSAGE, params);
    clientThread.send(gameMessage);
  }

  /**
   * method which connects the client to the server, if the client already runs a thread opens a new
   * socket with the ip address and the port of the server
   *
   * @return status (as boolean) if connection worked (true) or failed (false)
   */
  public boolean connectToServer() {
    if (clientThread != null) {
      isConnected = true;
      return true;
    }
    System.out.println("Establishing connection. Please wait ...");
    try {
      socket = new Socket(serverAddress, serverPort);
      System.out.println("Connected: " + socket);
      start(socket);
      isConnected = true;
      return true;
    } catch (IOException e) {
      isConnected = false;
      System.out.println("Unexpected exception: " + e.getMessage());
      return false;
    }
  }

  /**
   * starts a new Thread for the client with the socket (parameter)
   */
  private void start(Socket socket) throws IOException {
    clientThread = new ClientThread(this, socket, this.tempNickname);
    Timer timer = new Timer();
    timer.schedule(new TimerTask() {
      @Override
      public void run() {
        checkPingPong();
        activeLobbiesRequest();
        activeClientsRequest();
      }
    }, pingInterval, pingInterval);
  }

  /**
   * requesting the infos on all active Clients from the Server.
   */
  private void activeClientsRequest() {
    if (clientThread == null) {
      return;
    }
    long now = System.currentTimeMillis();
    System.out.println("active clients request... " + toReadableTime(now));
    IMessage requestMsg = CommandFactory.createMessage(GET_CLIENT_INFO, null);
    clientThread.send(requestMsg);
  }

  /**
   * requesting the infos on all active Lobbies from the Server.
   */
  private void activeLobbiesRequest() {
    if (clientThread == null) {
      return;
    }
    long now = System.currentTimeMillis();
    System.out.println("active lobbies request... " + toReadableTime(now));
    IMessage requestMsg = CommandFactory.createMessage(GET_LOBBY_INFO, null);
    clientThread.send(requestMsg);
  }

  /**
   * closes/stops the thread
   */
  void stopThread() {
    if (clientThread != null) {
      clientThread.interrupt();
      clientThread = null;
      clientGUI.disconnect();
    }
  }

  /**
   * the method changes the name of the client and adds the new name on the playerlist for the same
   * client
   *
   * @param name your new ingame name
   */
  public void changeName(String name) {
    ArrayList<String> params = new ArrayList<>();
    params.add(name);
    IMessage myNameMsg = CommandFactory.createMessage(MY_NICKNAME, params);
    clientThread.send(myNameMsg);
  }

  /**
   * sends a PING to the server to check if the Server connection is available.
   */
  private synchronized void checkPingPong() {
    if (clientThread == null) {
      return;
    }
    long now = System.currentTimeMillis();
    System.out.println("Ping check..." + toReadableTime(now));
    IMessage pingMsg = CommandFactory.createMessage(PING, null);
    if (now - clientThread.getLastContact() > 4 * pingInterval) {
      clientThread.close();
    } else if (now - clientThread.getLastContact() > pingInterval) {
      clientThread.send(pingMsg);
      System.out.println("Ping sent to server at:" + toReadableTime(now));
    }

  }

  /**
   * sends message that the game begins
   */
  void sendBeginGame() {
    System.out.println("Sending begin game message");
    IMessage beginGame = CommandFactory.createMessage(BEGIN_GAME, null);
    clientThread.send(beginGame);
  }

  /**
   * method which sends the chat message (private message) to the receiver
   *
   * @param receiver - player who receives the message
   * @param msg - message which was send
   */
  private void sendChatMessageToReceiver(String receiver, String msg) {
    System.out.println("Sending chat message: " + msg);
    ArrayList<String> params = new ArrayList<>();
    params.add(receiver);
    params.add(msg);
    IMessage gameMessage = CommandFactory.createMessage(SEND_TO_ADDRESS, params);
    clientThread.send(gameMessage);
  }

  /**
   * handles the change of the players direction, only changes if possible (if there is no wall)
   *
   * @param direction the direction the player wants to go
   */
  public void tryToChangeDirection(String direction) {
    System.out.println("Sending directionChange: " + direction);
    ArrayList<String> params = new ArrayList<>();
    params.add(direction);
    IMessage gameMessage = CommandFactory.createMessage(CHANGE_DIRECTION, params);
    clientThread.send(gameMessage);
  }

  /**
   * method which sends the message that a new player joined the lobby (in chat)
   *
   * @param lobbyName - name of the lobby connected to
   */
  void sendJoinLobbyMessage(String lobbyName) {
    lobbyName = lobbyName.replace(IMessage.separator, "_");
    System.out.println("Sending join lobby message: " + lobbyName);
    clientThread.setMyLobby(lobbyName);
    ArrayList<String> params = new ArrayList<>();
    params.add(lobbyName);
    IMessage gameMessage = CommandFactory.createMessage(JOIN_LOBBY, params);
    clientThread.send(gameMessage);
  }

  /**
   * sends the saved manual to Client
   */
  public void sendGetManual() {
    System.out.println(("Sending GET_MANUAL"));
    IMessage getManual = CommandFactory.createMessage(GET_MANUAL, null);
    clientThread.send(getManual);
  }

  /**
   * sends the saved highscore
   */
  public void sendGetHighScore() {
    System.out.println("Sending GET_HIGHSCORE");
    IMessage getHighScore = CommandFactory.createMessage(GET_HIGHSCORE, null);
    clientThread.send(getHighScore);
  }

  /**
   * converts the given time into a readable String.
   *
   * @param timeInms - time to convert in millisenconds
   * @return - converted time as a String
   */
  private String toReadableTime(long timeInms) {
    SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
    Date resultDate = new Date(timeInms);
    return sdf.format(resultDate);
  }

  /**
   * decides which case the send button is, sending in lobby or breadcasting
   *
   * "NotInLobby" is reserved as keyword
   */
  void handleSendRequest(String input) {
    if (isClientInLobby()) {
      String receiver = clientThread.getMyLobby();
      sendChatMessageToReceiver(receiver, input);
    } else {
      sendChatMessageToReceiver("NotInLobby", input);
    }
  }

  /**
   * handles the whisper message by splitting it in an array (head: receiver; body: info of
   * message)
   *
   * @param input the chat message (private message)
   */
  void handleWhisperRequest(String input) {
    String[] splitted = input.split(" ");
    String receiver = splitted[0];
    String msg = splitted[1];
    if (splitted.length >= 2) {
      for (int i = 2; i < splitted.length; i++) {
        msg = msg + " " + splitted[i];
      }
      sendChatMessageToReceiver(receiver, msg);
    }
  }

  /**
   * gets infos about the map, creates and paints it to the ClientGUI.
   *
   * @param mapMessage - message with infos about the map that should be created.
   * @throws Exception - throws exception if something goes wrong with the map-load.
   */
  public synchronized void handleMap(MapMessage mapMessage) throws Exception {
    map = new Map();
    String input = mapMessage.getParamValueSize();
    String[] parts = input.split(",");
    map.xSize = Integer.parseInt(parts[0]);
    map.ySize = Integer.parseInt(parts[1]);
    map.tSize = Integer.parseInt(parts[2].replace(";", ""));
    input = mapMessage.getParamValueWalls();
    String[] walls = input.split("!");
    for (String w : walls) {
      map.Walls.add(createWall(w));
    }
    input = mapMessage.getParamValueSpawn();
    String[] spawn = input.split("!");
    for (String s : spawn) {
      map.Spawn.add(createWall(s));
    }
    input = mapMessage.getParamValuePowerBalls();
    map.PowerBalls.addAll(createPowerBalls(input));
    input = mapMessage.getParamValueBalls();
    map.Balls.addAll(createBalls(input));
    input = mapMessage.getParamValueGate();
    map.gate = createGate(input);
    clientGUI.paintMap(map);
  }

  /**
   * handles the "decision" of which player is going to be which type (PacMan or Ghost?)
   *
   * @param paramValue information about the initialization of the players
   */
  public void handleActorInit(String paramValue) {
    String[] playerType = paramValue.split(";");
    for (String p : playerType) {
      String[] parts = p.split(",");
      if (parts[1].equals("PacMan")) {
        Point2D point2D = new Point2D.Float(java.lang.Float.parseFloat(parts[2]),
            java.lang.Float.parseFloat(parts[3]));
        actors.put(parts[0], new PacMan((Float) point2D, null, (Float) point2D));
      } else if (parts[1].equals("Ghost")) {
        Point2D point2D = new Point2D.Float(java.lang.Float.parseFloat(parts[3]),
            java.lang.Float.parseFloat(parts[4]));
        actors.put(parts[0], new Ghost((Float) point2D, null, (Float) point2D));
        ((Ghost) actors.get(parts[0])).color = GhostColor.valueOf(parts[2]);
      }
    }
  }

  /**
   * initialization of the game state
   *
   * @param paramValue information about the initialization of the game state
   */
  public void handleInitGameState(String paramValue) {
    String[] parts = paramValue.split(",");
    gameState = new GameState(Integer.parseInt(parts[2]), (Integer.parseInt(parts[1])));
    gameState.superPower = Integer.parseInt(parts[0]);
  }

  /**
   * handles the state of the game, containing scores and "superpower"
   *
   * @param paramValue information about the current game state
   */
  public void handleGameState(String paramValue) {
    String[] parts = paramValue.split(",");
    gameState.superPower = Integer.parseInt(parts[0]);
    gameState.scoreGhost = Integer.parseInt(parts[1]);
    gameState.scorePacMan = Integer.parseInt(parts[2]);
  }

  /**
   * handles the position and direction of the player on the field, gives infos to the client GUI
   *
   * @param parm information about the initialisation of the players
   */
  public void handleActorPos(String parm) {
    String[] playerType = parm.split(";");
    for (String p : playerType) {
      String[] parts = p.split(",");
      Player ply = actors.get(parts[0]);
      ply.direction = Direction.valueOf(parts[1]);
      Point2D point2D = new Point2D.Float(java.lang.Float.parseFloat(parts[2]),
          java.lang.Float.parseFloat(parts[3]));
      ply.setPosition((Float) point2D);
    }
  }

  /**
   * method which handles the removing of an object, which was taken buy a player from the field,
   * deletes the object from the client GUI
   *
   * @param paramValue information about the initialisation of the players
   */
  public synchronized void handleRemoveObject(String paramValue) {
    String[] parts = paramValue.split(",");
    Point2D.Float point = new Point2D.Float(java.lang.Float.parseFloat(parts[0]),
        java.lang.Float.parseFloat(parts[1]));
    Object o = map.getObject(point);
    if (o instanceof Ball) {
      map.removeBall((Ball) o);
    }
    if (o instanceof PowerBall) {
      map.removePowerBall((PowerBall) o);
    }
    clientGUI.paintMap(map);
  }

  /**
   * creates a gate on the map by analysing the line inputs it gets.
   *
   * @param line -  String that contains the infos about a map-line.
   * @return - returns the created gate.
   */
  private Gate createGate(String line) {
    int x = 0, y = 0;
    Point2D.Float point = null;
    String[] coordinated = line.split(",");
    x = (int) java.lang.Float.parseFloat(coordinated[0]);
    y = (int) java.lang.Float.parseFloat(coordinated[1].replace(";", ""));
    point = new Point2D.Float(x, y);
    Gate gate = new Gate(point);
    return gate;
  }

  /**
   * creates a ball on the map by analysing the line input it gets.
   *
   * @param line - String that contains the infos about a map-line.
   * @return - returns the created ball.
   */
  private List<Ball> createBalls(String line) {
    int x = 0, y = 0;
    Point2D.Float point = null;
    List<Ball> allBalls = new ArrayList<Ball>();
    String[] points = line.split(";");
    for (int i = 0; i < points.length; i++) {
      String[] coordinates = points[i].split(",");
      x = (int) java.lang.Float.parseFloat(coordinates[0]);
      y = (int) java.lang.Float.parseFloat(coordinates[1]);
      point = new Point2D.Float(x, y);
      Ball Ball = new Ball(point);
      allBalls.add(Ball);
    }
    return allBalls;
  }

  /**
   * creates a Powerball on the map by analysing the line input it gets.   * @param line
   *
   * @param line - String that contains the infos about a map-line.
   * @return - returns the created Powerball.
   */
  private List<PowerBall> createPowerBalls(String line) {
    int x = 0, y = 0;
    Point2D.Float point = null;
    List<PowerBall> allBalls = new ArrayList<PowerBall>();
    String[] points = line.split(";");
    for (int i = 0; i < points.length; i++) {
      String[] coordinates = points[i].split(",");
      x = (int) java.lang.Float.parseFloat(coordinates[0]);
      y = (int) java.lang.Float.parseFloat(coordinates[1]);
      ;
      point = new Point2D.Float(x, y);
      PowerBall powerBall = new PowerBall(point);
      allBalls.add(powerBall);
    }
    return allBalls;
  }

  /**
   * creates a Wall on the map by analysing the line input it gets.   * @param line
   *
   * @param line - String that contains the infos about a map-line.
   * @return - returns the created Wall.
   */
  private Wall createWall(String line) {
    int x = 0, y = 0;
    Point2D.Float point = null;
    List<Float> Points = new ArrayList<>();
    String[] pointcoordinates = line.split(";");
    List<String[]> coordinates = new ArrayList<String[]>();
    for (int i = 0; i < pointcoordinates.length; i++) {
      coordinates.add(pointcoordinates[i].split(","));
    }
    for (String[] strings : coordinates) {
      x = (int) java.lang.Float.parseFloat(strings[0]);
      y = (int) java.lang.Float.parseFloat(strings[1]);
      point = new Point2D.Float(x, y);
      Points.add(point);
    }
    Wall wall = new Wall(Points);
    return wall;
  }

  /**
   * handles the lobby response
   *
   * @param b status of lobby connection
   */
  public void forwardJoinLobbyResponse(boolean b) {
    clientGUI.handleLobbyResponse(b);
  }

  /**
   * handles the private message (whisper) for the GUI
   *
   * @param chatToMeMessage the private message
   */
  void forwardChatToMeMessage(ChatWithAddress chatToMeMessage) {
    clientGUI.handleChatToMeMessage(chatToMeMessage);
  }

  /**
   * handles the chat message for the GUI
   *
   * @param chatMessage the chat message
   */
  void forwardChat(ChatMessage chatMessage) {
    clientGUI.handleChat(chatMessage);
  }

  /**
   * handles the failure of a beginning game (less than 2 players in a game) for the GUI
   */
  public void forwardBeginFailed() {
    clientGUI.handleBeginFailed();
  }

  /**
   * handles the name input in the info for the GUI
   *
   * @param myName name of the client
   */
  void forwardName(String myName) {
    clientGUI.initName(myName);
  }

  /**
   * handles the lobby-info for the GUI
   */
  void forwardLobbyInfo(String serverInfo) {
    clientGUI.setLobbyInfo(serverInfo);
  }

  /**
   * handles the client-info for the GUI
   */
  void forwardClientInfo(String serverInfo) {
    clientGUI.setClientInfo(serverInfo);
  }

  /**
   * handles the visualization of the highscore for the GUI
   *
   * @param paramValue the highscore as string
   */
  public void forwardHighScore(String paramValue) {
    clientGUI.handleHighScore(paramValue);
  }

  /**
   * handles the manual for the GUI
   *
   * @param paramValue the text in the manual
   */
  public void forwardManual(String paramValue) {
    clientGUI.handleManual(paramValue);
  }

  public void handleGameFinish(GameFinishedMessage gameFinishedMessage) {
    clientGUI.setGameFinish(gameFinishedMessage.getAnswer());
  }

  /**
   * check method for the connection status
   *
   * @return connection status (true: connected; false: not connected)
   */
  public boolean isConnected() {
    return isConnected;
  }

  /**
   * method to proof if the client is in the lobby, checks status through a getter
   *
   * @return boolean for the status if player is in the lobby
   */
  boolean isClientInLobby() {
    if (this.clientThread.getMyLobby() != null) {
      return true;
    }
    return false;
  }

  /**
   * "getter" for client thread
   *
   * @return the thread of the client
   */
  public ClientThread getClientThread() {
    return clientThread;
  }

  /**
   * "getter" for the lobby of a client
   *
   * @return the lobbyname of the client
   */
  public String getLobby() {
    return clientThread.getMyLobby();
  }

  /**
   * method which returns all the actors/players in the game
   *
   * @return the players
   */
  public HashMap<String, Player> getActors() {
    return actors;
  }

  /**
   * "getter" fot the current game state
   *
   * @return the current game state
   */
  public GameState getGameState() {
    return gameState;
  }
}