package pacManReloaded.client;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.Point2D;
import java.awt.geom.Point2D.Float;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Clip;
import javax.swing.BorderFactory;
import javax.swing.GroupLayout;
import javax.swing.GroupLayout.Alignment;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.LayoutStyle.ComponentPlacement;
import javax.swing.UIManager;
import org.lwjgl.LWJGLException;
import org.lwjgl.input.Keyboard;
import org.lwjgl.opengl.Display;
import org.lwjgl.opengl.DisplayMode;
import org.lwjgl.opengl.GL11;
import org.newdawn.slick.TrueTypeFont;
import org.newdawn.slick.opengl.Texture;
import org.newdawn.slick.opengl.TextureLoader;
import org.newdawn.slick.util.ResourceLoader;
import pacManReloaded.common.Lock;
import pacManReloaded.common.messages.ChatMessage;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.model.Ball;
import pacManReloaded.common.model.Gate;
import pacManReloaded.common.model.Ghost;
import pacManReloaded.common.model.Map;
import pacManReloaded.common.model.PacMan;
import pacManReloaded.common.model.PowerBall;
import pacManReloaded.common.model.Sound;
import pacManReloaded.common.model.Wall;
import pacManReloaded.server.GameState;
import pacManReloaded.server.Player;


public class ClientGUI extends JFrame {

  private JPanel rootPanel;
  private JTextArea lobbyTextArea;
  private JTextArea clientsTextArea;
  private JButton btnSend;
  private JButton btnBroadcast;
  private JButton btnWhisper;
  private JButton btnChangeName;
  private JTextArea chatTextArea;
  private JTextField chatInputTextField;
  private JTextArea playerInfoTextArea;
  private JTextArea gameMasterTextArea;
  private JButton btnConnect;
  private JButton btnDisconnect;
  private JButton btnJoin;
  private JButton btnBegin;
  private JButton btnHighscore;
  private JButton btnManual;
  private Thread glThread;
  private Lock lock = new Lock();
  private Sound sound;


  final int FRAMES_PER_SECOND = 20;
  private volatile boolean isRunning = false;

  private TrueTypeFont font;
  private TrueTypeFont font2;
  private boolean antiAlias = true;

  private Client client;
  private String myName;
  private Map map;

  private Texture tex;
  private float x, y;

  HashMap<String, Texture> textures;

  private final int tSize = 20;
  private int WIDTH;
  private int HEIGHT;

  private boolean finish;
  private String finishResult;

  private static boolean closeRequested = false;
  private final static AtomicReference<Dimension> newCanvasSize = new AtomicReference<Dimension>();

  /**
   * Initialize the contents of the frame.
   */
  private ClientGUI() {
    sound = new Sound();
    Font f = new Font("Arial", Font.TRUETYPE_FONT, 13);
    Color yellow = new Color(253, 240, 0);
    Color blue = new Color(63, 72, 203);

    setBounds(100, 100, 1160, 940);
    setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

    UIManager.put("TabbedPane.contentAreaColor", Color.BLACK);
    UIManager.put("TabbedPane.selected", Color.BLACK);
    Insets insets = UIManager.getInsets("TabbedPane.contentBorderInsets");
    insets.top = -1;
    UIManager.put("TabbedPane.contentBorderInsets", insets);
    insets.bottom = -1;
    UIManager.put("TabbedPane.contentBorderInsets", insets);
    insets.left = -1;
    UIManager.put("TabbedPane.contentBorderInsets", insets);
    insets.right = -1;
    UIManager.put("TabbedPane.contentBorderInsets", insets);

    rootPanel = new JPanel();
    rootPanel.setBackground(Color.BLACK);
    rootPanel.setBorder(BorderFactory.createLineBorder(blue));
    GroupLayout groupLayout = new GroupLayout(this.getContentPane());
    groupLayout.setHorizontalGroup(groupLayout.createParallelGroup(Alignment.LEADING)
        .addComponent(rootPanel, GroupLayout.DEFAULT_SIZE, 1160, Short.MAX_VALUE));
    groupLayout.setVerticalGroup(groupLayout.createParallelGroup(Alignment.LEADING)
        .addComponent(rootPanel, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 880,
            Short.MAX_VALUE));

    JPanel controlPanel = new JPanel();
    controlPanel.setBackground(Color.BLACK);

    JPanel playingFieldPanel = new JPanel();
    playingFieldPanel.setBackground(Color.BLACK);

    this.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        setVisible(false);
        dispose();  //canvas's removeNotify() will be called
      }
    });
    final Canvas canvas = new Canvas() {
      @Override
      public void addNotify() {
        super.addNotify();
        startGL();
      }

      @Override
      public void removeNotify() {
        stopGL();
        super.removeNotify();
      }
    };
    canvas.addComponentListener(new ComponentAdapter() {
      @Override
      public void componentResized(ComponentEvent e) {
        newCanvasSize.set(canvas.getSize());
      }
    });

    this.addWindowFocusListener(new WindowAdapter() {
      @Override
      public void windowGainedFocus(WindowEvent e) {
        canvas.requestFocusInWindow();
      }
    });

    this.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        closeRequested = true;
      }
    });

    canvas.setPreferredSize(new Dimension(660, 880));
    canvas.setIgnoreRepaint(true);

    try {
      Display.setParent(canvas);
      Display.setVSyncEnabled(true);
    } catch (LWJGLException e) {
      //handle exception
      e.printStackTrace();
    }

    playingFieldPanel.add(canvas);

    JPanel infoPanel = new JPanel();
    infoPanel.setBackground(Color.BLACK);
    GroupLayout gl_rootPanel = new GroupLayout(rootPanel);
    gl_rootPanel.setHorizontalGroup(gl_rootPanel.createParallelGroup(Alignment.LEADING).addGroup(
        gl_rootPanel.createSequentialGroup()
            .addComponent(controlPanel, GroupLayout.DEFAULT_SIZE, 220, Short.MAX_VALUE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(playingFieldPanel, GroupLayout.DEFAULT_SIZE, 660, Short.MAX_VALUE)
            .addGap(13).addComponent(infoPanel, GroupLayout.PREFERRED_SIZE, 280, Short.MAX_VALUE)));
    gl_rootPanel.setVerticalGroup(gl_rootPanel.createParallelGroup(Alignment.LEADING)
        .addComponent(controlPanel, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 880,
            Short.MAX_VALUE).addGroup(gl_rootPanel.createSequentialGroup()
            .addComponent(playingFieldPanel, GroupLayout.DEFAULT_SIZE, 880, Short.MAX_VALUE)
            .addContainerGap())
        .addComponent(infoPanel, GroupLayout.DEFAULT_SIZE, 880, Short.MAX_VALUE));

    JPanel chatPanel = new JPanel();
    chatPanel.setBackground(Color.BLACK);

    JPanel lobby_clientPanel = new JPanel();
    lobby_clientPanel.setBackground(Color.BLACK);
    GroupLayout gl_infoPanel = new GroupLayout(infoPanel);
    gl_infoPanel.setHorizontalGroup(gl_infoPanel.createParallelGroup(Alignment.LEADING).addGroup(
        gl_infoPanel.createSequentialGroup().addGroup(
            gl_infoPanel.createParallelGroup(Alignment.TRAILING)
                .addComponent(lobby_clientPanel, Alignment.LEADING, 0, 0, Short.MAX_VALUE)
                .addComponent(chatPanel, Alignment.LEADING, GroupLayout.PREFERRED_SIZE, 278,
                    Short.MAX_VALUE)).addGap(2)));
    gl_infoPanel.setVerticalGroup(gl_infoPanel.createParallelGroup(Alignment.TRAILING).addGroup(
        gl_infoPanel.createSequentialGroup()
            .addComponent(chatPanel, GroupLayout.DEFAULT_SIZE, GroupLayout.DEFAULT_SIZE,
                Short.MAX_VALUE).addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(lobby_clientPanel, GroupLayout.DEFAULT_SIZE, 347, Short.MAX_VALUE)));

    JScrollPane scrollPane_2 = new JScrollPane();
    scrollPane_2.setBorder(BorderFactory.createLineBorder(blue));
    scrollPane_2.setBackground(Color.BLACK);
    GroupLayout gl_lobby_clientPanel = new GroupLayout(lobby_clientPanel);
    gl_lobby_clientPanel.setHorizontalGroup(
        gl_lobby_clientPanel.createParallelGroup(Alignment.LEADING).addGroup(
            gl_lobby_clientPanel.createSequentialGroup()
                .addComponent(scrollPane_2, GroupLayout.DEFAULT_SIZE, 284, Short.MAX_VALUE)
                .addContainerGap()));
    gl_lobby_clientPanel.setVerticalGroup(
        gl_lobby_clientPanel.createParallelGroup(Alignment.LEADING).addGroup(
            gl_lobby_clientPanel.createSequentialGroup()
                .addComponent(scrollPane_2, GroupLayout.DEFAULT_SIZE, 332, Short.MAX_VALUE)
                .addContainerGap()));

    JTabbedPane tabbedPane = new JTabbedPane(JTabbedPane.TOP);
    tabbedPane.setForeground(yellow);
    tabbedPane.setBackground(Color.DARK_GRAY);
    getContentPane().setBackground(Color.BLACK);
    tabbedPane.setBorder(BorderFactory.createLineBorder(blue));
    scrollPane_2.setViewportView(tabbedPane);
    lobbyTextArea = new JTextArea();
    lobbyTextArea.setBorder(BorderFactory.createLineBorder(blue));
    lobbyTextArea.setDisabledTextColor(yellow);
    lobbyTextArea.setBackground(Color.BLACK);
    lobbyTextArea.setEnabled(false);
    lobbyTextArea.setEditable(false);
    lobbyTextArea.setFont(f);
    tabbedPane.addTab("Lobby", null, lobbyTextArea, null);
    clientsTextArea = new JTextArea();
    clientsTextArea.setBorder(BorderFactory.createLineBorder(blue));
    clientsTextArea.setBackground(Color.BLACK);
    clientsTextArea.setDisabledTextColor(yellow);
    clientsTextArea.setEnabled(false);
    clientsTextArea.setEditable(false);
    clientsTextArea.setFont(f);
    tabbedPane.addTab("Clients", null, clientsTextArea, null);
    lobby_clientPanel.setLayout(gl_lobby_clientPanel);

    JScrollPane scrollPane_1 = new JScrollPane();
    scrollPane_1.setBorder(BorderFactory.createLineBorder(blue));
    scrollPane_1.setBackground(Color.BLACK);

    JPanel chat_functionPanel = new JPanel();
    chat_functionPanel.setBackground(Color.BLACK);
    GroupLayout gl_chatPanel = new GroupLayout(chatPanel);
    gl_chatPanel.setHorizontalGroup(gl_chatPanel.createParallelGroup(Alignment.TRAILING).addGroup(
        gl_chatPanel.createSequentialGroup().addGroup(
            gl_chatPanel.createParallelGroup(Alignment.TRAILING)
                .addComponent(chat_functionPanel, Alignment.LEADING, GroupLayout.PREFERRED_SIZE,
                    263, Short.MAX_VALUE)
                .addComponent(scrollPane_1, GroupLayout.DEFAULT_SIZE, 263, Short.MAX_VALUE))
            .addGap(15)));
    gl_chatPanel.setVerticalGroup(gl_chatPanel.createParallelGroup(Alignment.TRAILING).addGroup(
        gl_chatPanel.createSequentialGroup().addContainerGap()
            .addComponent(scrollPane_1, GroupLayout.DEFAULT_SIZE, 329, Short.MAX_VALUE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(chat_functionPanel, GroupLayout.PREFERRED_SIZE, 110, Short.MAX_VALUE)
            .addGap(4)));

    chatInputTextField = new JTextField();
    chatInputTextField.setBackground(Color.black);
    chatInputTextField.setForeground(yellow);
    chatInputTextField.setBorder(BorderFactory.createLineBorder(blue));
    chatInputTextField.setColumns(10);

    btnSend = new JButton("Send");
    btnSend.setBackground(Color.BLACK);
    btnSend.setForeground(yellow);
    btnSend.setBorder(BorderFactory.createLineBorder(blue));
    btnSend.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        sendButtonGetClicked();
      }
    });

    btnWhisper = new JButton("Whisper");
    btnWhisper.setBackground(Color.BLACK);
    btnWhisper.setForeground(yellow);
    btnWhisper.setBorder(BorderFactory.createLineBorder(blue));
    btnWhisper.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        whisperButtonGetClicked();
      }
    });

    btnBroadcast = new JButton("Broadcast");
    btnBroadcast.setBackground(Color.BLACK);
    btnBroadcast.setForeground(yellow);
    btnBroadcast.setBorder(BorderFactory.createLineBorder(blue));
    btnBroadcast.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        broadcastButtonGetClicked();
      }
    });
    GroupLayout gl_chat_functionPanel = new GroupLayout(chat_functionPanel);
    gl_chat_functionPanel.setHorizontalGroup(
        gl_chat_functionPanel.createParallelGroup(Alignment.LEADING).addGroup(
            gl_chat_functionPanel.createSequentialGroup().addGroup(
                gl_chat_functionPanel.createParallelGroup(Alignment.LEADING).addGroup(
                    gl_chat_functionPanel.createSequentialGroup().addComponent(chatInputTextField)
                        .addPreferredGap(ComponentPlacement.RELATED)
                        .addComponent(btnSend, GroupLayout.DEFAULT_SIZE, 108, Short.MAX_VALUE))
                    .addGroup(gl_chat_functionPanel.createSequentialGroup()
                        .addComponent(btnWhisper, GroupLayout.DEFAULT_SIZE, 144, Short.MAX_VALUE)
                        .addPreferredGap(ComponentPlacement.RELATED)
                        .addComponent(btnBroadcast, GroupLayout.DEFAULT_SIZE, 110, Short.MAX_VALUE)
                        .addPreferredGap(ComponentPlacement.RELATED))).addGap(0)));
    gl_chat_functionPanel.setVerticalGroup(
        gl_chat_functionPanel.createParallelGroup(Alignment.LEADING).addGroup(
            gl_chat_functionPanel.createSequentialGroup().addGap(6).addGroup(
                gl_chat_functionPanel.createParallelGroup(Alignment.BASELINE)
                    .addComponent(btnSend, GroupLayout.DEFAULT_SIZE, 37, Short.MAX_VALUE)
                    .addComponent(chatInputTextField)).addPreferredGap(ComponentPlacement.UNRELATED)
                .addGroup(gl_chat_functionPanel.createParallelGroup(Alignment.BASELINE)
                    .addComponent(btnWhisper, GroupLayout.DEFAULT_SIZE, 36, Short.MAX_VALUE)
                    .addComponent(btnBroadcast, GroupLayout.DEFAULT_SIZE, 36, Short.MAX_VALUE))
                .addGap(22)));
    chat_functionPanel.setLayout(gl_chat_functionPanel);

    JTabbedPane tabbedPane_2 = new JTabbedPane(JTabbedPane.TOP);
    tabbedPane_2.setForeground(yellow);
    tabbedPane_2.setBackground(Color.DARK_GRAY);
    scrollPane_1.setViewportView(tabbedPane_2);

    chatTextArea = new JTextArea();
    chatTextArea.setBorder(BorderFactory.createLineBorder(blue));
    chatTextArea.setDisabledTextColor(yellow);
    chatTextArea.setBackground(Color.BLACK);
    chatTextArea.setEnabled(false);
    chatTextArea.setEditable(false);
    chatTextArea.setFont(f);
    chatTextArea.setDisabledTextColor(yellow);
    tabbedPane_2.addTab("Chat", null, chatTextArea, null);
    chatPanel.setLayout(gl_chatPanel);
    infoPanel.setLayout(gl_infoPanel);

    JPanel buttonPanel = new JPanel();
    buttonPanel.setBackground(Color.BLACK);

    JPanel notePanel = new JPanel();
    notePanel.setBackground(Color.BLACK);

    JScrollPane scrollPane = new JScrollPane();
    scrollPane.setBorder(BorderFactory.createLineBorder(blue));
    scrollPane.setBackground(Color.BLACK);
    GroupLayout gl_notePanel = new GroupLayout(notePanel);
    gl_notePanel.setHorizontalGroup(gl_notePanel.createParallelGroup(Alignment.LEADING)
        .addGroup(Alignment.TRAILING, gl_notePanel.createSequentialGroup().addContainerGap()
            .addComponent(scrollPane, GroupLayout.DEFAULT_SIZE, 181, Short.MAX_VALUE)));
    gl_notePanel.setVerticalGroup(gl_notePanel.createParallelGroup(Alignment.LEADING).addGroup(
        gl_notePanel.createSequentialGroup().addContainerGap()
            .addComponent(scrollPane, GroupLayout.DEFAULT_SIZE, 377, Short.MAX_VALUE)));

    JTabbedPane tabbedPane_1 = new JTabbedPane(JTabbedPane.TOP);
    tabbedPane_1.setForeground(yellow);
    tabbedPane_1.setBackground(Color.DARK_GRAY);
    scrollPane.setViewportView(tabbedPane_1);

    gameMasterTextArea = new JTextArea();
    gameMasterTextArea.setEnabled(false);
    gameMasterTextArea.setEditable(false);
    gameMasterTextArea.setBorder(BorderFactory.createLineBorder(blue));
    gameMasterTextArea.setDisabledTextColor(yellow);
    gameMasterTextArea.setBackground(Color.BLACK);
    gameMasterTextArea.setFont(f);
    tabbedPane_1.addTab("GameMaster", null, gameMasterTextArea, null);

    playerInfoTextArea = new JTextArea();
    playerInfoTextArea.setEnabled(false);
    playerInfoTextArea.setEditable(false);
    playerInfoTextArea.setBorder(BorderFactory.createLineBorder(blue));
    playerInfoTextArea.setDisabledTextColor(yellow);
    playerInfoTextArea.setBackground(Color.BLACK);
    playerInfoTextArea.setFont(f);
    tabbedPane_1.addTab("PlayerInfo", null, playerInfoTextArea, null);
    notePanel.setLayout(gl_notePanel);
    GroupLayout gl_controlPanel = new GroupLayout(controlPanel);
    gl_controlPanel.setHorizontalGroup(gl_controlPanel.createParallelGroup(Alignment.TRAILING)
        .addGroup(gl_controlPanel.createSequentialGroup().addGroup(
            gl_controlPanel.createParallelGroup(Alignment.TRAILING)
                .addComponent(notePanel, GroupLayout.DEFAULT_SIZE, 197, Short.MAX_VALUE)
                .addComponent(buttonPanel, GroupLayout.DEFAULT_SIZE, 197, Short.MAX_VALUE))
            .addGap(2)));
    gl_controlPanel.setVerticalGroup(gl_controlPanel.createParallelGroup(Alignment.LEADING)
        .addGroup(Alignment.TRAILING, gl_controlPanel.createSequentialGroup()
            .addComponent(notePanel, GroupLayout.DEFAULT_SIZE, 393, Short.MAX_VALUE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(buttonPanel, GroupLayout.DEFAULT_SIZE, 422, Short.MAX_VALUE)));

    btnConnect = new JButton("Connect");
    btnConnect.setBackground(Color.BLACK);
    btnConnect.setForeground(yellow);
    btnConnect.setBorder(BorderFactory.createLineBorder(blue));
    btnConnect.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent arg0) {
        connectButtonGetClicked();
      }
    });

    btnDisconnect = new JButton("Disconnect");
    btnDisconnect.setBackground(Color.BLACK);
    btnDisconnect.setForeground(yellow);
    btnDisconnect.setBorder(BorderFactory.createLineBorder(blue));
    btnDisconnect.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        disconnectButtonGetClicked();
      }
    });

    btnJoin = new JButton("Join");
    btnJoin.setBackground(Color.BLACK);
    btnJoin.setForeground(yellow);
    btnJoin.setBorder(BorderFactory.createLineBorder(blue));
    btnJoin.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        joinButtonGetClicked();
      }
    });

    btnBegin = new JButton("Begin");
    btnBegin.setBackground(Color.BLACK);
    btnBegin.setForeground(yellow);
    btnBegin.setBorder(BorderFactory.createLineBorder(blue));
    btnBegin.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        beginButtonGetClicked();
      }
    });

    btnHighscore = new JButton("Highscore");
    btnHighscore.setBackground(Color.BLACK);
    btnHighscore.setForeground(yellow);
    btnHighscore.setBorder(BorderFactory.createLineBorder(blue));
    btnHighscore.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        highScoreButtonGetClicked();
      }
    });

    btnManual = new JButton("Manual");
    btnManual.setBackground(Color.BLACK);
    btnManual.setForeground(yellow);
    btnManual.setBorder(BorderFactory.createLineBorder(blue));
    btnManual.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        manualButtonGetClicked();
      }
    });

    btnChangeName = new JButton("Change Name");
    btnChangeName.setBackground(Color.BLACK);
    btnChangeName.setForeground(yellow);
    btnChangeName.setBorder(BorderFactory.createLineBorder(blue));
    btnChangeName.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent arg0) {
        changeNameButtonGetClicked();
      }
    });
    GroupLayout gl_buttonPanel = new GroupLayout(buttonPanel);
    gl_buttonPanel.setHorizontalGroup(gl_buttonPanel.createParallelGroup(Alignment.TRAILING)
        .addGroup(gl_buttonPanel.createSequentialGroup().addGap(33).addGroup(
            gl_buttonPanel.createParallelGroup(Alignment.LEADING)
                .addComponent(btnChangeName, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 154,
                    Short.MAX_VALUE)
                .addComponent(btnManual, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 154,
                    Short.MAX_VALUE)
                .addComponent(btnHighscore, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 154,
                    Short.MAX_VALUE)
                .addComponent(btnJoin, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 154,
                    Short.MAX_VALUE)
                .addComponent(btnDisconnect, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 154,
                    Short.MAX_VALUE)
                .addComponent(btnConnect, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 154,
                    Short.MAX_VALUE)
                .addComponent(btnBegin, Alignment.TRAILING, GroupLayout.DEFAULT_SIZE, 154,
                    Short.MAX_VALUE)).addContainerGap()));
    gl_buttonPanel.setVerticalGroup(gl_buttonPanel.createParallelGroup(Alignment.LEADING).addGroup(
        gl_buttonPanel.createSequentialGroup().addContainerGap()
            .addComponent(btnConnect, GroupLayout.DEFAULT_SIZE, 51, Short.MAX_VALUE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(btnDisconnect, GroupLayout.DEFAULT_SIZE, 40, GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(btnJoin, GroupLayout.PREFERRED_SIZE, 39, GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(btnBegin, GroupLayout.PREFERRED_SIZE, 41, GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(btnHighscore, GroupLayout.PREFERRED_SIZE, 41, GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(btnManual, GroupLayout.PREFERRED_SIZE, 40, GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(ComponentPlacement.RELATED)
            .addComponent(btnChangeName, GroupLayout.DEFAULT_SIZE, 50, Short.MAX_VALUE)
            .addGap(70)));
    buttonPanel.setLayout(gl_buttonPanel);
    controlPanel.setLayout(gl_controlPanel);
    rootPanel.setLayout(gl_rootPanel);
    this.getContentPane().setLayout(groupLayout);
    initButtons();
  }

  /**
   * starts the visualization with LWJGL
   */
  private void startGL() {
    textures = new HashMap<String, Texture>();
    WIDTH = 33 * tSize;
    HEIGHT = 44 * tSize;
    glThread = new Thread(new Runnable() {
      @Override
      public void run() {
        isRunning = true;
        try {
          Display.setDisplayMode(new DisplayMode(WIDTH, HEIGHT));
          Display.create();
        } catch (LWJGLException e) {
          //handle exception
          e.printStackTrace();
        }
        GL11.glMatrixMode(GL11.GL_PROJECTION);
        GL11.glLoadIdentity();
        GL11.glOrtho(0, WIDTH, HEIGHT, 0, 1, -1);
        GL11.glMatrixMode(GL11.GL_MODELVIEW);
        GL11.glEnable(GL11.GL_TEXTURE_2D);
        GL11.glEnable(GL11.GL_BLEND);
        GL11.glBlendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);
        Dimension newDim;
        while (isRunning) {
          newDim = newCanvasSize.getAndSet(null);
          if (newDim != null) {
            GL11.glViewport(0, 0, newDim.width, newDim.height);
          }
          GL11.glClear(GL11.GL_COLOR_BUFFER_BIT);
          try {
            if (map != null) {
              paint();
            }
          } catch (IOException e) {
            e.printStackTrace();
          } catch (InterruptedException e) {
            e.printStackTrace();
          }
          Display.update();
          Display.sync(FRAMES_PER_SECOND);
          checkKeyboard();
        }

        Display.destroy();
      }
    }, "LWJGL Thread");

    glThread.start();
  }

  /**
   * stops the visualization of LWJGL
   */
  private void stopGL() {
    isRunning = false;
    try {
      glThread.join();
    } catch (InterruptedException e) {
      //handle exception
      e.printStackTrace();
    }
  }

  /**
   * paints all the game objects wiht LWJGL
   *
   * @throws IOException IOException with null as error message
   * @throws InterruptedException exception for the failure by interruption
   */
  public synchronized void paint() throws IOException, InterruptedException {
    GL11.glClear(GL11.GL_COLOR_BUFFER_BIT);
    drawWalls(map.Walls);
    intersectionWalls(map.Walls);
    drawWalls(map.Spawn);
    intersectionWalls(map.Spawn);
    drawGate(map.gate);
    lock.lock();
    synchronized (map.Balls) {
      try {
        drawPowerballs(map.PowerBalls);
        drawBall(map.Balls);
      } finally {
        lock.unlock();
      }
    }
    drawActors(client.getActors(), client.getGameState());
    if (finish) {
      handleGameFinish(finishResult);
    }
    initFont();
    render(client.getGameState().scorePacMan, client.getGameState().scoreGhost);
  }

  /**
   * draws the PacMans and the Ghosts in the GUI, sets the game state
   *
   * @param actorsPos the actor (Pacman / Ghost)
   * @param gameState the state of the game
   * @throws IOException IOException with null as error message
   */
  private void drawActors(HashMap<String, Player> actorsPos, GameState gameState)
      throws IOException {
    Set<String> keys = actorsPos.keySet();
    for (String k : keys) {
      Player actor = actorsPos.get(k);
      if (actor instanceof PacMan) {
        drawPacMan((PacMan) actor);
      }
      if (actor instanceof Ghost) {
        drawGhost((Ghost) actor, gameState);
      }
    }

  }

  /**
   * draws the PacMan by loading the png of the resources directory
   *
   * @param pacMan the PacMan
   * @throws IOException IOException with null as error message
   */
  private void drawPacMan(PacMan pacMan) throws IOException {
    tex = quickLoad("PacMan");
    float x = pacMan.position.x;
    float y = pacMan.position.y;
    switch (pacMan.direction) {
      case left:
        drawQuadTex(tex, x, y, tSize, tSize, -180.0f);
        break;
      case right:
        drawQuadTex(tex, x, y, tSize, tSize, 0.0f);
        break;

      case up:
        drawQuadTex(tex, x, y, tSize, tSize, -90.0f);
        break;

      case down:
        drawQuadTex(tex, x, y, tSize, tSize, 90.0f);
        break;
      default:
        break;
    }
  }

  /**
   * draws the ghost by loading the png of the resources directory
   *
   * @param ghost the ghost
   * @throws IOException IOException with null as error message
   */
  private void drawGhost(Ghost ghost, GameState gameState) throws IOException {
    Texture tex = null;
    if (gameState.superPower > 0) {
      tex = quickLoad("Ghost_fear");
    } else {
      switch (ghost.color) {
        case blue:
          tex = quickLoad("Ghost_blue");
          break;
        case red:
          tex = quickLoad("Ghost_red");
          break;
        case pink:
          tex = quickLoad("Ghost_pink");
          break;
        case yellow:
          tex = quickLoad("Ghost_yellow");
          break;
        default:
          break;
      }
    }

    float x = ghost.position.x;
    float y = ghost.position.y;

    drawQuadTex(tex, x, y, tSize, tSize);
  }

  /**
   * draws the gate by loading the png of the resources directory
   *
   * @param gate the gate
   * @throws IOException IOException with null as error message
   */
  private void drawGate(Gate gate) throws IOException {
    tex = quickLoad("Tor");
    float x = gate.position.x;
    float y = gate.position.y;
    drawQuadTex(tex, x, y, tSize, tSize);

  }

  /**
   * draws the balls by loading the png of the resources directory
   *
   * @param pBalls the balls in a list
   * @throws IOException IOException with null as error message
   */
  private synchronized void drawBall(List<Ball> pBalls) throws IOException {
    tex = quickLoad("Kleiner_Punkt");
    for (int i = 0; i < pBalls.size(); i++) {
      float x = pBalls.get(i).position.x;
      float y = pBalls.get(i).position.y;
      drawQuadTex(tex, x, y, tSize, tSize);
    }
  }

  /**
   * draws the power balls by loading the png of the resources directory
   *
   * @param pBalls the power balls
   * @throws IOException IOException with null as error message
   */
  private synchronized void drawPowerballs(List<PowerBall> pBalls) throws IOException {
    tex = quickLoad("Grosser_Punkt");
    for (PowerBall powerBall : pBalls) {
      float x = powerBall.position.x;
      float y = powerBall.position.y;
      drawQuadTex(tex, x, y, tSize, tSize);
    }
  }

  /**
   * draws the walls by loading the png of the resources directory
   *
   * @param Walls the walls
   * @throws IOException IOException with null as error message
   */
  private void drawWalls(List<Wall> Walls) throws IOException {
    for (Wall w : Walls) {
      Point2D.Float[] pointArray = new Point2D.Float[w.points.size()];
      for (int i = 0; i < pointArray.length; i++) {
        pointArray[i] = w.points.get(i);
      }
      for (int i = 0; i < pointArray.length; i++) {
        if (i < pointArray.length - 1) {
          if (pointArray[i].x + 1 == pointArray[i + 1].x) {
            tex = quickLoad("Horizontal");
            x = (float) pointArray[i].x;
            y = (float) pointArray[i].y;
            drawQuadTex(tex, x, y, tSize, tSize);
          } else if (pointArray[i].y + 1 == pointArray[i + 1].y) {
            tex = quickLoad("Vertical");
            x = (float) pointArray[i].x;
            y = (float) pointArray[i].y;
            drawQuadTex(tex, x, y, tSize, tSize);
          }
        }
        if (i > 0) {
          if (pointArray[i].x - 1 == pointArray[i - 1].x) {
            tex = quickLoad("Horizontal");
            x = (float) pointArray[i].x;
            y = (float) pointArray[i].y;
            drawQuadTex(tex, x, y, tSize, tSize);
          } else if (pointArray[i].y - 1 == pointArray[i - 1].y) {
            tex = quickLoad("Vertical");
            x = (float) pointArray[i].x;
            y = (float) pointArray[i].y;
            drawQuadTex(tex, x, y, tSize, tSize);
          }
        }
      }
    }
  }

  /**
   * handles the selection of the matching png for the corners, so if 3 walls intersect, there has
   * to be a different graphic as if only 2 walls connect
   */
  private void checkCorners(Float point3, List<Float> points) throws IOException {
    for (Point2D.Float point : points) {
      if ((point.getX() == point3.getX() + 1 && point.getY() == point3.getY())) {
        for (Point2D.Float point1 : points) {
          if ((point1.getX() == point3.getX() && point1.getY() == point3.getY() + 1)) {
            tex = quickLoad("Rechts_Unten_Kurve");
            drawQuadTex(tex, point3.x, point3.y, tSize, tSize);
          }
          if (point1.getX() == point3.getX() && point1.getY() + 1 == point3.getY()) {
            tex = quickLoad("Oben_Rechts_Kurve");
            drawQuadTex(tex, point3.x, point3.y, tSize, tSize);
          }

        }
      }
      if ((point.getX() == point3.getX() - 1 && point.getY() == point3.getY())) {
        for (Point2D.Float point2 : points) {
          if (point2.getX() == point3.getX() && point2.getY() + 1 == point3.getY()) {
            tex = quickLoad("Links_Oben_Kurve");
            drawQuadTex(tex, point3.x, point3.y, tSize, tSize);
          }
          if (point2.getX() == point3.getX() && point2.getY() - 1 == point3.getY()) {
            tex = quickLoad("Links_Unten_Kurve");
            drawQuadTex(tex, point3.x, point3.y, tSize, tSize);
          }
        }
      }

    }
  }

  /**
   * handles the visualization of the intersection of walls; choses different image depending on the
   * amount of walls intersections
   *
   * @param Walls the walls
   * @throws IOException IOException with null as error message
   */
  private void intersectionWalls(List<Wall> Walls) throws IOException {
    List<Point2D.Float> points = new ArrayList<>();
    for (Wall w : Walls) {
      Point2D.Float[] pointArray = new Point2D.Float[w.points.size()];
      for (int i = 0; i < pointArray.length; i++) {
        pointArray[i] = w.points.get(i);
        points.add(pointArray[i]);
      }
    }
    for (Point2D.Float point : points) {
      int j = 0;
      int i = j;
      while (i < points.size()) {
        if (point == points.get(i)) {
          checkCorners(point, points);
        }
        i++;
      }
      j++;

    }

  }

  /**
   * loads the game textures of the file
   *
   * @param path the path to the directory of the file
   * @param fileType type of the file
   * @return the texture of the file
   * @throws IOException IOException with null as error message
   */
  private Texture loadTexture(String path, String fileType) throws IOException {
    if (textures.containsKey(path)) {
      return textures.get(path);
    }
    System.out.println(path);

    Texture tex = null;
    InputStream in = ResourceLoader.getResourceAsStream(path);
    tex = (Texture) TextureLoader.getTexture(fileType, in);
    textures.put(path, tex);
    return tex;
  }

  /**
   * method to load the pngs of the resource folders by their names
   *
   * @param name name of the png
   * @return the texture of the png
   * @throws IOException IOException with null as error message
   */
  private Texture quickLoad(String name) throws IOException {
    Texture tex = null;
    tex = loadTexture("resources/" + name + ".png", "PNG");
    return tex;
  }

  private void drawQuadTex(Texture tex, float x, float y, float width, float height) {
    tex.bind();
    GL11.glTranslatef(x * tSize, y * tSize, 0);
    GL11.glBegin(GL11.GL_QUADS);
    GL11.glTexCoord2f(0, 0);
    GL11.glVertex2f(0, 0);
    GL11.glTexCoord2f(1, 0);
    GL11.glVertex2f(width, 0);
    GL11.glTexCoord2f(1, 1);
    GL11.glVertex2f(width, height);
    GL11.glTexCoord2f(0, 1);
    GL11.glVertex2f(0, height);
    GL11.glEnd();
    GL11.glLoadIdentity();
  }

  /**
   * initializes the font of the text
   */
  public void initFont() {
    Font awtFont = new Font("Times New Roman", Font.BOLD, 18);
    font = new TrueTypeFont(awtFont, antiAlias);
    try {
      InputStream inputStream = ResourceLoader.getResourceAsStream("myfont.ttf");

      Font awtFont2 = Font.createFont(Font.TRUETYPE_FONT, inputStream);
      awtFont2 = awtFont2.deriveFont(20f);
      font2 = new TrueTypeFont(awtFont2, antiAlias);
    } catch (Exception e) {

    }
  }

  /**
   * renders the scores of the players on the top left (PacMan) and top right (Ghost) side
   *
   * @param scorePacMan score of the team PacMan
   * @param scoreGhost score of the team Ghost
   */
  public void render(int scorePacMan, int scoreGhost) {
    org.newdawn.slick.Color.white.bind();

    font.drawString(20, 20, "Score PacMan: " + Integer.toString(scorePacMan),
        org.newdawn.slick.Color.white);
    font.drawString(520, 20, "Score Ghost: " + Integer.toString(scoreGhost),
        org.newdawn.slick.Color.white);
  }

  /**
   * renders the finish game message, prints out if you won or lost and the score
   *
   * @param score the score of the player
   * @param text the result message
   */
  private void gameOverText(int score, String text) {
    render(score, text);
  }

  /**
   * renders the scores of both teams on the chosen location
   *
   * @param score the score of the team
   * @param text the team name
   */
  public void render(int score, String text) {
    org.newdawn.slick.Color.black.bind();
    font.drawString(280, 400, text, org.newdawn.slick.Color.white);
    font.drawString(280, 430, "Score: " + Integer.toString(score), org.newdawn.slick.Color.white);
  }

  private void drawQuadTex(Texture tex, float x, float y, float width, float height,
      float rotation) {
    tex.bind();
    GL11.glTranslatef(x * tSize, y * tSize, 0);
    GL11.glTranslatef(tSize / 2, tSize / 2, 0);
    GL11.glRotatef(rotation, 0, 0, 1);
    GL11.glTranslatef(-tSize / 2, -tSize / 2, 0);

    GL11.glBegin(GL11.GL_QUADS);
    GL11.glTexCoord2f(0, 0);
    GL11.glVertex2f(0, 0);
    GL11.glTexCoord2f(1, 0);
    GL11.glVertex2f(width, 0);
    GL11.glTexCoord2f(1, 1);
    GL11.glVertex2f(width, height);
    GL11.glTexCoord2f(0, 1);
    GL11.glVertex2f(0, height);
    GL11.glEnd();
    GL11.glLoadIdentity();
  }

  /**
   * sets the enability of the buttons after the server started, only connect button enabled until
   * you are connected to the server
   */
  private void initButtons() {
    btnConnect.setEnabled(true);
    btnJoin.setEnabled(false);
    btnBegin.setEnabled(false);
    btnSend.setEnabled(false);
    btnDisconnect.setEnabled(false);
    btnBroadcast.setEnabled(false);
    btnWhisper.setEnabled(false);
    btnHighscore.setEnabled(false);
    btnManual.setEnabled(false);
    btnChangeName.setEnabled(false);
  }

  /**
   * initializes the palyers name
   */
  void initName(String myName) {
    this.myName = myName;
    updatePlayerInfo();
  }

  /**
   * updates the infos of each players, sets the text in the info box for name, connection and
   * joined lobby
   */
  void updatePlayerInfo() {
    playerInfoTextArea.setText("");
    playerInfoTextArea.append("Name: " + myName + System.lineSeparator());
    playerInfoTextArea.append("Connection: " + client.isConnected() + System.lineSeparator());
    playerInfoTextArea.append("Lobby: " + client.getLobby() + System.lineSeparator());
  }

  /**
   * Create the application.
   */
  static ClientGUI initGUI() {
    ClientGUI frame = new ClientGUI();
    frame.setTitle("PacMan Reloaded");
    frame.setVisible(true);
    return frame;
  }

  /**
   * sends the manual if its button gets clicked
   */
  private void manualButtonGetClicked() {
    client.sendGetManual();
  }

  /**
   * sends the highscore if its button gets clicked
   */
  private void highScoreButtonGetClicked() {
    client.sendGetHighScore();
  }

  /**
   * handles the usage of the change name button, only works if connected to server
   */
  private void changeNameButtonGetClicked() {
    if (client.isConnected()) {
      String name = JOptionPane.showInputDialog(rootPanel, "Name: ", null);
      if (name != null && !name.equals("")) {
        client.changeName(name);
      }
    } else {
      gameMasterTextArea.append("Du musst dich mit dem Server verbinden!" + System.lineSeparator());
    }
  }

  /**
   * let the game begin when the begin button gets clicked
   */
  private void beginButtonGetClicked() {
    if (client.isClientInLobby()) {
      client.sendBeginGame();
      playerInfoTextArea.setText(playerInfoTextArea.getText() + "READY");
      btnBegin.setEnabled(false);
    } else {
      gameMasterTextArea.append(
          "Die Lobby-Anfrage hat nicht geklappt.\nBitte w\u00e4hle eine andere Lobby!" + System
              .lineSeparator());
      btnBegin.setEnabled(false);
    }
  }

  /**
   * handles the joining for the lobby if the button gets clicked
   */
  private void joinButtonGetClicked() {
    if (client.isConnected()) {
      String lobby = JOptionPane.showInputDialog(rootPanel, "Lobby: ", null);
      if (lobby != null && !lobby.equals("")) {
        client.sendJoinLobbyMessage(lobby);
        finish = false;
        finishResult = "";
      }
    } else {
      gameMasterTextArea.append("Du musst dich mit dem Server verbinden!" + System.lineSeparator());
    }
  }

  /**
   * handles the clicking of the sending button (--> message getting send)
   */
  private void sendButtonGetClicked() {
    String input = chatInputTextField.getText();
    if (input.length() > 0) {
      chatInputTextField.setText("");
      client.handleSendRequest(input);
    }
  }

  /**
   * handles the clicking of the sending button (--> whisper getting send)
   */
  private void whisperButtonGetClicked() {
    String message = chatInputTextField.getText();
    String receiver = JOptionPane.showInputDialog(rootPanel, "Message should be send to?");
    if (message.length() > 0 && receiver.length() > 0) {
      chatInputTextField.setText("");
      client.handleWhisperRequest(receiver + " " + message);
    }
  }

  /**
   * handles the clicking of the sending button (--> broadcasting message)
   */
  private void broadcastButtonGetClicked() {
    String input = chatInputTextField.getText();
    if (input.length() > 0) {
      chatInputTextField.setText("");
      client.sendChatMessage(input);
    }
  }

  /**
   * connects the client to the server if the connection button gets clicked
   */
  private void connectButtonGetClicked() {
    if (!client.isConnected()) {
      if (client.connectToServer()) {
        JOptionPane.showMessageDialog(rootPanel, "Connection established!");
        gameMasterTextArea.setText("");
        btnDisconnect.setEnabled(true);
        btnJoin.setEnabled(true);
        btnConnect.setEnabled(false);
        btnSend.setEnabled(true);
        btnBroadcast.setEnabled(true);
        btnWhisper.setEnabled(true);
        chatInputTextField.setEnabled(true);
        btnHighscore.setEnabled(true);
        btnManual.setEnabled(true);
        btnChangeName.setEnabled(true);
        updatePlayerInfo();
      } else {
        JOptionPane.showMessageDialog(rootPanel,
            "Cannot established connection to " + client.serverAddress + " at port "
                + client.serverPort);
      }
    }
  }

  /**
   * disconnects the client from the server if the dc-button gets clicked
   */
  private void disconnectButtonGetClicked() {
    client.disconnectFromServer();
    disconnect();
  }

  /**
   * "setter" for the client
   *
   * @param client the client
   */
  public void setClient(Client client) {
    this.client = client;
  }

  /**
   * shows a client if he is disconnected by writing it down in the info box in the bottom on the
   * right side
   */
  public void disconnect() {
    playerInfoTextArea.setText("");
    initButtons();
    gameMasterTextArea.setText("");
    gameMasterTextArea.append("Sie sind disconnected.");
    chatInputTextField.setEnabled(false);
  }

  /**
   * handles the failure of the game start, if you try to start a game while there are not enough
   * player (4) in the lobby, shows it in the info box in the bottom on the right side
   */
  void handleBeginFailed() {
    gameMasterTextArea
        .append("Du bist momentan alleine in der Lobby.\nWarte auf weiterne Spieler.");
    btnBegin.setEnabled(true);
  }

  /**
   * handles the response of the lobby, if you joined a lobby you are able to click the begin game
   * button; handles the error for the failure of joining a lobby
   */
  void handleLobbyResponse(boolean response) {
    if (response) {
      btnBegin.setEnabled(true);
      btnJoin.setEnabled(false);
      updatePlayerInfo();
    } else {
      gameMasterTextArea.append(
          "Die Lobby-Anfrage hat nicht geklappt.\nBitte w\u00e4hle eine andere Lobby!" + System
              .lineSeparator());
    }
  }

  /**
   * handles the chat message, appends to the text area what a player sended and who it was
   *
   * @param chatMessage the chat message
   */
  public void handleChat(ChatMessage chatMessage) {
    chatTextArea
        .append(chatMessage.getFrom() + ": " + chatMessage.getInfo() + System.lineSeparator());
    playSoundChat();
  }

  /**
   * handles the chat message, appends to the text area what a player sended and who it was
   *
   * @param chatToMeMessage the whisper message to "me/you"
   */
  public void handleChatToMeMessage(ChatWithAddress chatToMeMessage) {
    chatTextArea.append(
        chatToMeMessage.getAddress() + ": " + chatToMeMessage.getInfo() + System.lineSeparator());
    playSoundChat();
  }

  /**
   * "setter" for the highscore
   *
   * @param paramValue the highscore as a string
   */
  public void handleHighScore(String paramValue) {
    InformationGUI.initInformationGUI(paramValue, this.size(), "High Score");
  }

  /**
   * handles the manual in the GUI
   *
   * @param paramValue the string of the manual
   */
  public void handleManual(String paramValue) {
    InformationGUI.initInformationGUI(paramValue, this.size(), "Manual");
  }

  /**
   * handles the message for the finished game, shows if you won or lost the game by checking the
   * scores
   */
  public void handleGameFinish(String answer) {
    String points;
    if (answer.equals("WON")) {
      if (client.getGameState().scorePacMan > client.getGameState().scoreGhost) {
        gameOverText(client.getGameState().scorePacMan, "You Win");

      }
      if (client.getGameState().scorePacMan <= client.getGameState().scoreGhost) {
        gameOverText(client.getGameState().scoreGhost, "You Win");

      }
    }
    if (answer.equals("LOSE")) {
      if (client.getGameState().scorePacMan <= client.getGameState().scoreGhost) {

        gameOverText(client.getGameState().scorePacMan, "You Lose");
      }
      if (client.getGameState().scorePacMan > client.getGameState().scoreGhost) {

        gameOverText(client.getGameState().scoreGhost, "You Lose");
      }
    }
  }

  /**
   * loads and plays the sound for the game finish message
   */
  public void playSoundFinishedGame() {
    try {
      File soundFile = new File("resources/personen_yeeaaa.wav");
      playSound(soundFile);
    } catch (Exception e) {
      System.out.println("Sound could not be loaded");
    }
  }

  /**
   * method which okays the sound of the reading file
   *
   * @param soundFile the name of the Soundfile to play
   */
  public void playSound(File soundFile) {
    try {
      AudioInputStream audioIn = AudioSystem.getAudioInputStream(soundFile);
      Clip clip = AudioSystem.getClip();
      clip.open(audioIn);
      clip.start();
    } catch (Exception e) {
      System.out.println("Sound could not be played");
    }
  }

  /**
   * loads and plays the sound for an incoming chat message
   */
  public void playSoundChat() {
    try {
      File soundFile = new File("resources/personen_husten.wav");
      playSound(soundFile);
    } catch (Exception e) {
      System.out.println("Sound could not be loaded");
    }
  }

  /**
   * sets finish state to true, so you can see who won the game and you are able to join lobbies
   * again (also activates a finish sound)
   */
  public void setGameFinish(String answer) {
    finish = true;
    finishResult = answer;
    playSoundFinishedGame();
    btnJoin.setEnabled(true);
  }

  /**
   * "setter" for the info of the server
   *
   * @param serverInfo server info as string
   */
  public void setLobbyInfo(String serverInfo) {
    lobbyTextArea.setText(serverInfo);
  }

  /**
   * "setter" for the info of the server
   *
   * @param serverInfo server info as string
   */
  public void setClientInfo(String serverInfo) {
    clientsTextArea.setText(serverInfo);
  }

  /**
   * paints the map on the clients GUI
   *
   * @param map the map
   */
  public void paintMap(Map map) {
    updatePlayerInfo();
    this.map = map;
  }

  /**
   * checks the input of the keyboard, sends the information of the direction change to the clients
   */
  public void checkKeyboard() {
    if (Keyboard.isKeyDown(Keyboard.KEY_LEFT)) {
      client.tryToChangeDirection("left");
    }
    if (Keyboard.isKeyDown(Keyboard.KEY_RIGHT)) {
      client.tryToChangeDirection("right");
    }
    if (Keyboard.isKeyDown(Keyboard.KEY_UP)) {
      client.tryToChangeDirection("up");
    }
    if (Keyboard.isKeyDown(Keyboard.KEY_DOWN)) {
      client.tryToChangeDirection("down");
    }
  }
}