package pacManReloaded.client;

import static pacManReloaded.common.Command.MY_NICKNAME;
import static pacManReloaded.common.Command.PONG;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import pacManReloaded.common.CommandFactory;
import pacManReloaded.common.messages.ChatMessage;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.messages.GameFinishedMessage;
import pacManReloaded.common.messages.IMessage;
import pacManReloaded.common.messages.MapMessage;
import pacManReloaded.common.messages.NoParameterMessage;
import pacManReloaded.common.messages.SingleStringParameterMessage;

public class ClientProtocol {

  /**
   * method which makes to time to a readable string
   *
   * @param timeInms - time in milliseconds
   * @return - time as a string
   */
  private String toReadableTime(long timeInms) {
    SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
    Date resultDate = new Date(timeInms);
    return sdf.format(resultDate);
  }

  /**
   * method which handles the different message types through a switch case operation.
   *
   * @param thread - thread of the client
   * @param gameMsg - the message which has to be handled
   */
  void handleMsg(ClientThread thread, IMessage gameMsg) {
    switch (gameMsg.getType()) {
      /**
       *  Gets a Message from the Client and handles it via the Thread.
       */
      case MESSAGE:
        ChatMessage chatMessage = (ChatMessage) gameMsg;
        thread.handleChat(chatMessage);
        break;
      case SEND_TO_ADDRESS:
        ChatWithAddress specificMessage = (ChatWithAddress) gameMsg;
        thread.handleChatToMeMessage(specificMessage);
        break;
      case GAME_FINISHED:
        GameFinishedMessage gameFinishedMessage = (GameFinishedMessage) gameMsg;
        thread.handleGameFinish(gameFinishedMessage);
        break;
      /**
       * sends a Ping to the Server to check the connection.
       */
      case PING:
        IMessage pongMsg = new NoParameterMessage(PONG);
        thread.send(pongMsg);
        System.out.println("Pong sent as answer at: " + toReadableTime(System.currentTimeMillis()));
        break;

      /**
       * sends a Pong to the Server to finish the connection test made by the client successfully.
       */
      case PONG:
        thread.updateLastContact();
        System.out.println(
            "Received a Pong from Server at: " + toReadableTime(System.currentTimeMillis())
        );
        break;
      /**
       * closes the Thread of the Client.
       */
      case CLOSE:
        thread.close();
        break;
      /**
       * gets the nickname from the client via the Thread and distributes it.
       */
      case GET_NICKNAME:
        String myName = thread.getMyName();
        ArrayList<String> params = new ArrayList<>();
        params.add(myName);
        IMessage myNameMsg = CommandFactory.createMessage(MY_NICKNAME, params);
        thread.send(myNameMsg);
        break;
      case HIGHSCORE: {
        SingleStringParameterMessage hScoreMsg = (SingleStringParameterMessage) gameMsg;
        thread.handleServerHighScore(hScoreMsg.getParamValue());
        break;
      }
      case MAP: {
        MapMessage mapMessage = (MapMessage) gameMsg;
        thread.handleMap(mapMessage);
        break;
      }
      case MANUAL: {
        SingleStringParameterMessage manualMsg = (SingleStringParameterMessage) gameMsg;
        thread.handleManual(manualMsg.getParamValue());
        break;
      }
      case LOBBY_INFO: {
        SingleStringParameterMessage infoMsg = (SingleStringParameterMessage) gameMsg;
        thread.handleServerLobbyInfo(infoMsg.getParamValue());
        break;
      }
      case CLIENT_INFO: {
        SingleStringParameterMessage infoMsg = (SingleStringParameterMessage) gameMsg;
        thread.handleServerClientInfo(infoMsg.getParamValue());
        break;
      }
      /**
       * sets the new Nickname of the Client based on the incoming Parameter.
       */
      case SET_NICKNAME:
        SingleStringParameterMessage newName = (SingleStringParameterMessage) gameMsg;
        thread.setMyName(newName.getParamValue());
        break;
      /**
       * Is called when a Unknown Message type occurs.
       * Informs via the Protocol output that a undefined command was used.
       */case JOIN_LOBBY_RESPONSE:
        SingleStringParameterMessage joinLobby = (SingleStringParameterMessage) gameMsg;
        thread.handleJoinLobbyResponse(joinLobby.getParamValue());
        break;
      case INIT_PLAYER:
        SingleStringParameterMessage initPlayer = (SingleStringParameterMessage) gameMsg;
        thread.handleActorInit(initPlayer.getParamValue());
        break;
      case ACTOR_POSITION:
        SingleStringParameterMessage actorPos = (SingleStringParameterMessage) gameMsg;
        thread.handleActorPos(actorPos.getParamValue());
        break;
      case INIT_GAMESTATE:
        SingleStringParameterMessage initGameState = (SingleStringParameterMessage) gameMsg;
        thread.handleInitGameState(initGameState.getParamValue());
        break;
      case GAMESTATE:
        SingleStringParameterMessage gameState = (SingleStringParameterMessage) gameMsg;
        thread.handleGameState(gameState.getParamValue());
        break;
      case REMOVE_OBJECT:
        SingleStringParameterMessage removeObjectMsg = (SingleStringParameterMessage) gameMsg;
        thread.handleRemoveObject(removeObjectMsg.getParamValue());
        break;
      case UNKNOWN_MESSAGE:
        System.out.println("This Command is not defined: " + gameMsg);
        break;
      default:
        System.out.println("Unknown Command: " + gameMsg);
        break;
    }
  }

}