package pacManReloaded.client;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.Date;
import pacManReloaded.common.CommandFactory;
import pacManReloaded.common.JoinLobbyResult;
import pacManReloaded.common.messages.ChatMessage;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.messages.GameFinishedMessage;
import pacManReloaded.common.messages.IMessage;
import pacManReloaded.common.messages.MapMessage;

public class ClientThread extends Thread {

  private Socket socket;
  private Client client;
  private BufferedReader streamIn = null;
  private BufferedWriter streamOut = null;
  private ClientProtocol protocol = new ClientProtocol();


  private long lastContact;
  private String nickname;
  private String myLobby;

  /**
   * Constructor of ClientThread
   *
   * @param _client name of the client
   * @param _socket name of the socket
   * @param myName nickname of the player
   */
  public ClientThread(Client _client, Socket _socket, String myName) {
    client = _client;
    socket = _socket;
    nickname = myName;
    myLobby = null;
    open();
    start();
    lastContact = System.currentTimeMillis();
  }

  /**
   * run method listens to the input stream, processes the data that it receives
   */
  public void run() {
    while (true) {
      try {
        String msg = streamIn.readLine();
        if (msg != null) {
          System.out.println(
              "Message received: " + msg + " at: " + toReadableTime(System.currentTimeMillis()));
          IMessage gameMsg = CommandFactory.parseMessage(msg);
          protocol.handleMsg(this, gameMsg);
        } else {
          System.out.println("End of stream reached!");
          break;
        }
      } catch (IOException ioe) {
        System.out.println("Listening error: " + ioe.getMessage());
        break;
      }
    }
    close();
  }

  /**
   * method to send the message
   *
   * @param msg message to be send
   */
  public void send(IMessage msg) {
    try {
      streamOut.write(msg + System.lineSeparator());
      streamOut.flush();
    } catch (IOException ioe) {
      System.out.println("ERROR sending: " + ioe.getMessage());
    }
  }

  /**
   * opens the InputStream of the ClientThread over Buffered reader. Then opens the Outputstream of
   * the ClientThread over a Buffered writer.
   */
  private void open() {
    try {
      streamIn = new BufferedReader(new InputStreamReader(socket.getInputStream(), "utf-8"));
      streamOut = new BufferedWriter(new OutputStreamWriter(socket.getOutputStream(), "utf-8"));
    } catch (IOException ioe) {
      System.out.println("Error getting input stream: " + ioe);
      client.stopThread();
    }
  }

  /**
   * closes the Thread - disconnects the Client from the server
   */
  public void close() {
    System.out.println("Goodbye.");
    clearInfo();
    try {
      if (socket != null) {
        socket.close();
        socket = null;
      }
      if (streamIn != null) {
        streamIn.close();
        streamIn = null;
      }
      if (streamOut != null) {
        streamOut.close();
        streamOut = null;
      }
      client.stopThread();
    } catch (IOException ioe) {
      System.out.println("Error closing input stream: " + ioe);
    }
  }

  /**
   * clears info of the client (sets nickname and Lobby status to null)
   */
  private void clearInfo() {
    nickname = null;
    lastContact = 0;
  }

  /**
   * method which makes to time to a readable string
   *
   * @param timeInms time in milliseconds
   * @return time as a string
   */
  private String toReadableTime(long timeInms) {
    SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
    Date resultDate = new Date(timeInms);
    return sdf.format(resultDate);
  }

  /**
   * handles the chat message, decides which type of message you want to send
   *
   * @param chatMessage message you want to send
   */
  public void handleChat(ChatMessage chatMessage) {
    client.forwardChat(chatMessage);
    System.out
        .println("Received messages '" + chatMessage.getInfo() + "' from " + chatMessage.getFrom());
  }

  /**
   * handles the messages of the map, containing walls, balls, ...
   *
   * @param mapMessage information about the map
   */
  public void handleMap(MapMessage mapMessage) {
    try {
      client.handleMap(mapMessage);
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * method which handles the position and direction of the players
   *
   * @param parm information of the players position
   */
  public void handleActorPos(String parm) {
    client.handleActorPos(parm);
  }

  /**
   * handles the "decision" of which player is going to be which type (PacMan or Ghost?)
   *
   * @param paramValue information about the initialization of the players
   */
  public void handleActorInit(String paramValue) {
    client.handleActorInit(paramValue);
  }

  /**
   * handles the message, which was send to you (private message)
   *
   * @param chatToMeMessage message which you ("me") receive by another client/player
   */
  public void handleChatToMeMessage(ChatWithAddress chatToMeMessage) {
    client.forwardChatToMeMessage(chatToMeMessage);
    System.out.println(
        "Received chat to me message '" + chatToMeMessage.getInfo() + "' from " + chatToMeMessage
            .getAddress());
  }

  /**
   * handles the info of the server, and prints it
   *
   * @param serverInfo info of the server
   */
  public void handleServerLobbyInfo(String serverInfo) {
    System.out.println("Server info...");
    serverInfo = serverInfo.replace(IMessage.separator, System.lineSeparator());
    client.forwardLobbyInfo(serverInfo);
    System.out.println(serverInfo);
  }

  /**
   * handles the info of the server, and prints it
   *
   * @param serverInfo info of the server
   */
  public void handleServerClientInfo(String serverInfo) {
    System.out.println("Server info...");
    serverInfo = serverInfo.replace(IMessage.separator, System.lineSeparator());
    client.forwardClientInfo(serverInfo);
    System.out.println(serverInfo);
  }

  /**
   * handles to response of the lobby to your client
   *
   * @param lobbyResponseResult response of the lobby to the client
   */
  public void handleJoinLobbyResponse(String lobbyResponseResult) {
    try {
      JoinLobbyResult lobbyState = JoinLobbyResult.valueOf(lobbyResponseResult);
      switch (lobbyState) {
        case Created:
        case Joined:
          client.forwardJoinLobbyResponse(true);
          System.out.println("Joined lobby: " + lobbyResponseResult);
          break;
        case Rejected:
        case AlreadyPlaying:
          client.forwardJoinLobbyResponse(false);
          System.out.println("Cannot join lobby " + this.myLobby + ": " + lobbyResponseResult);
          this.myLobby = null;
          break;
      }
    } catch (IllegalArgumentException ex) {
      System.out.println("Unknown lobby response: " + lobbyResponseResult);
      this.myLobby = null;
    }
  }

  /**
   * handles the reading of the highscore file by the server
   *
   * @param paramValue the highscore as a string
   */
  public void handleServerHighScore(String paramValue) {
    System.out.println("Server info...");
    paramValue = paramValue.replace(IMessage.separator, System.lineSeparator());
    client.forwardHighScore(paramValue);
    System.out.println(paramValue);
  }

  /**
   * handles the manual for the client
   *
   * @param paramValue the manual as string
   */
  public void handleManual(String paramValue) {
    paramValue = paramValue.replace(IMessage.separator, System.lineSeparator());
    client.forwardManual(paramValue);
  }

  /**
   * initialization of the game state
   *
   * @param paramValue information about the initialization of the game state
   */
  public void handleInitGameState(String paramValue) {
    client.handleInitGameState(paramValue);
  }

  /**
   * handles the state of the game, containing scores and "superpower"
   *
   * @param paramValue information about the current game state
   */
  public void handleGameState(String paramValue) {
    client.handleGameState(paramValue);
  }

  /**
   * method which handles the removing of an object, which was taken buy a player from the field,
   * deletes the object from the client GUI
   *
   * @param paramValue information about the initialisation of the players
   */
  public void handleRemoveObject(String paramValue) {
    client.handleRemoveObject(paramValue);
  }

  /**
   * handles the starting game failed message for the client
   */
  public void handleBeginFailed() {
    client.forwardBeginFailed();
  }

  public void handleGameFinish(GameFinishedMessage gameFinishedMessage) {
    client.handleGameFinish(gameFinishedMessage);
  }

  /**
   * updates the last contact of the client to the server in milliseconds
   */
  public void updateLastContact() {
    this.lastContact = System.currentTimeMillis();
  }

  /**
   * "getter" for your name
   *
   * @return your nickname
   */
  public String getMyName() {
    return nickname;
  }

  /**
   * "getter" for last contact to the server
   *
   * @return time of the last contact to the server
   */
  public long getLastContact() {
    return lastContact;
  }

  /**
   * sets the name of the client
   *
   * @param myName nickname of the client
   */
  public void setMyName(String myName) {
    this.nickname = myName;
    client.forwardName(myName);
  }

  /**
   * "getter" for your lobby
   *
   * @return lobby you are connected to
   */
  public String getMyLobby() {
    return myLobby;
  }

  /**
   * "setter" for your lobby
   *
   * @param myLobby lobby you want to be connected to
   */
  public void setMyLobby(String myLobby) {
    this.myLobby = myLobby;
  }

}