package pacManReloaded.common;

import static pacManReloaded.common.Command.UNKNOWN_MESSAGE;

import java.util.ArrayList;
import pacManReloaded.common.messages.ChatMessage;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.messages.GameFinishedMessage;
import pacManReloaded.common.messages.IMessage;
import pacManReloaded.common.messages.MapMessage;
import pacManReloaded.common.messages.NoParameterMessage;
import pacManReloaded.common.messages.SingleStringParameterMessage;

public class CommandFactory {

  /**
   * method which parses the message
   *
   * @param msg message which gets parsed
   * @return the type of the message
   */
  public static IMessage parseMessage(String msg) {
    if (msg == null || msg.length() == 0) {
      return new NoParameterMessage(UNKNOWN_MESSAGE);
    }

    int index = msg.indexOf(IMessage.separator);
    String header = (index > 0) ? msg.substring(0, index) : msg;
    try {
      Command msgType = Command.getEnum(header);
      switch (msgType) {
        case MESSAGE: {
          return new ChatMessage(msg.substring(index + 1));
        }
        case SEND_TO_ADDRESS: {
          return new ChatWithAddress(msg.substring(index + 1));
        }
        case GET_NICKNAME: {
          return new NoParameterMessage(msgType);
        }
        case GET_LOBBY_INFO: {
          return new NoParameterMessage(msgType);
        }
        case LOBBY_INFO: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case GET_CLIENT_INFO: {
          return new NoParameterMessage(msgType);
        }
        case CLIENT_INFO: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case PING: {
          return new NoParameterMessage(msgType);
        }
        case PONG: {
          return new NoParameterMessage(msgType);
        }
        case MANUAL: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case MAP: {
          int indexToSplit = 0;
          String[] parts = new String[6];
          String toSplit = msg.substring(index + 1);
          for (int i = 0; i < 6; i++) {
            if (i < 5) {
              indexToSplit = toSplit.indexOf(IMessage.separator);
              parts[i] = toSplit.substring(0, indexToSplit);
              toSplit = toSplit.substring(indexToSplit + 1);
            } else {
              parts[5] = toSplit;
            }
          }
          return new MapMessage(msgType, parts[0],
              parts[1], parts[2], parts[3],
              parts[4], parts[5]);
        }
        case GET_MANUAL: {
          return new NoParameterMessage(msgType);
        }
        case BEGIN_GAME: {
          return new NoParameterMessage(msgType);
        }
        case HIGHSCORE: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case GET_HIGHSCORE: {
          return new NoParameterMessage(msgType);
        }
        case JOIN_LOBBY: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case GAME_FINISHED: {
          return new GameFinishedMessage(msg.substring(index + 1));
        }
        case BEGIN_FAILED: {
          return new NoParameterMessage(msgType);
        }
        case JOIN_LOBBY_RESPONSE: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case ACTOR_POSITION: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));

        }
        case CLOSE: {
          return new NoParameterMessage(msgType);
        }
        case MY_NICKNAME: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case REMOVE_OBJECT: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case GAMESTATE: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case INIT_GAMESTATE: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case CHANGE_DIRECTION: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case INIT_PLAYER: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        case SET_NICKNAME: {
          return new SingleStringParameterMessage(msgType, msg.substring(index + 1));
        }
        default: {
          System.out.println("Cannot parse messages: " + msg);
          return new NoParameterMessage(UNKNOWN_MESSAGE);
        }
      }
    } catch (IllegalArgumentException ex) {
      System.out.println("Unknown message " + header);
      System.out.println(ex.getMessage());
      return new NoParameterMessage(UNKNOWN_MESSAGE);
    }
  }

  /**
   * method which creates the message
   *
   * @param msgType type of the message
   * @param params the split String in an ArrayList
   * @return type of message
   */
  public static IMessage createMessage(Command msgType, ArrayList<String> params) {
    switch (msgType) {
      case GET_NICKNAME: {
        return new NoParameterMessage(msgType);
      }
      case PING: {
        return new NoParameterMessage(msgType);
      }
      case PONG: {
        return new NoParameterMessage(msgType);
      }
      case MY_NICKNAME: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case INIT_PLAYER: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case GET_LOBBY_INFO: {
        return new NoParameterMessage(msgType);
      }
      case LOBBY_INFO: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case GET_CLIENT_INFO: {
        return new NoParameterMessage(msgType);
      }
      case CLIENT_INFO: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case ACTOR_POSITION: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case CHANGE_DIRECTION: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case REMOVE_OBJECT: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case GAMESTATE: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case INIT_GAMESTATE: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case BEGIN_GAME: {
        return new NoParameterMessage(msgType);
      }
      case GET_HIGHSCORE: {
        return new NoParameterMessage(msgType);
      }
      case HIGHSCORE: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case GET_MANUAL: {
        return new NoParameterMessage(msgType);
      }
      case MANUAL: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case MAP: {
        assert params.size() > 0;
        return new MapMessage(msgType, params.get(0), params.get(1), params.get(2), params.get(3),
            params.get(4), params.get(5));
      }
      case JOIN_LOBBY: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case JOIN_LOBBY_RESPONSE: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case GAME_FINISHED: {
        assert params.size() > 1;
        return new GameFinishedMessage(params.get(0), params.get(1));
      }
      case BEGIN_FAILED: {
        return new NoParameterMessage(msgType);
      }
      case SET_NICKNAME: {
        assert params.size() > 0;
        return new SingleStringParameterMessage(msgType, params.get(0));
      }
      case CLOSE: {
        return new NoParameterMessage(msgType);
      }
      case MESSAGE: {
        assert params.size() > 1;
        return new ChatMessage(params.get(0), params.get(1));
      }
      case SEND_TO_ADDRESS: {
        assert params.size() > 1;
        return new ChatWithAddress(params.get(0), params.get(1));
      }
      default:
        return new NoParameterMessage(UNKNOWN_MESSAGE);
    }
  }

}