package pacManReloaded.server;

import static pacManReloaded.server.GameStatus.initialized;
import static pacManReloaded.server.GameStatus.playing;
import static pacManReloaded.server.GameStatus.sendingFinishInfo;

import java.awt.geom.Point2D;
import java.awt.geom.Point2D.Float;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import pacManReloaded.common.model.Ball;
import pacManReloaded.common.model.Ghost;
import pacManReloaded.common.model.INavigator;
import pacManReloaded.common.model.Map;
import pacManReloaded.common.model.PacMan;
import pacManReloaded.common.model.PowerBall;
import pacManReloaded.common.model.Wall;


public class GameLogic implements INavigator {

  public GameState gameState;
  public Map map;
  public ArrayList<Player> players;
  private Server server;

  /**
   * Constructor of the Class, starts a game with the players and server as parameters
   *
   * @param players the players who will be in the game as arraylist
   * @param server the server where the game will be played on
   */
  public GameLogic(ArrayList<Player> players, Server server) {
    try {
      map = new Map();
    } catch (Exception e) {
      e.printStackTrace();
    }
    this.players = players;
    this.server = server;
    gameState = new GameState(0, 0);
    loadMap("Map.txt");
    setGameStatus(playing);
    for (Player p : players) {
      p.sendInitPlayer(players);
      p.sendInitGameState(gameState);
      p.sendMap(map);
    }
  }

  /**
   * Dummy-Constructor for Unit tests
   */
  public GameLogic() {
    gameState = new GameState(0, 0);
    try {
      map = new Map();
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * loads the map of the directory
   *
   * @param path the path to the file of the map
   */
  private void loadMap(String path) {
    try {
      GameLoader loader = new GameLoader(this, players);
      loader.load(path, map, players);
      createBalls();
    } catch (Exception e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  /**
   * allows all actors to perform their next step, in addition, calls the operation
   * checkSuperPower() and detectCollision().
   */
  public void nextMove() {
    checkSuperPower();
    for (Player actor : players) {
      actor.nextMove();
      actor.sendActorPosition(players);
      actor.sendGameState(gameState);
    }
    detectCollisions();
  }

  /**
   * list of the positions of each actor in the game
   *
   * @return a list of all actors position
   */
  private List<Point2D> actorPosition() {
    List<Point2D> pos = new ArrayList<>();
    for (Player actor : players) {
      pos.add(actor.position);
    }
    return pos;
  }

  /**
   * this method finds out of the next step of an actor is a collision with another object/wall
   */
  public void detectCollisions() {
    for (Player p : players) {
      if (p instanceof Ghost && gameState.superPower > 0) {
        Object o = getObjectAtPosition(p.position);
        if (o instanceof PowerBall) {
          collisionWithPowerballGhost((PowerBall) o);
        } else if (o instanceof Ball) {
          collisionWithBallGhost((Ball) o);
        }
      }
      if (p instanceof PacMan && gameState.superPower == 0) {
        Object o = getObjectAtPosition(p.position);
        if (o instanceof PowerBall) {
          collisionWithPowerballPacMan((PowerBall) o);
        } else if (o instanceof Ball) {
          collisionWithBallPacMan((Ball) o);
        }

      }
      detectActorsCollisions(p);
    }

  }

  /**
   * detects the collisions of players with each others,
   *
   * @param p the player
   */
  private void detectActorsCollisions(Player p) {
    for (Player actor : players) {
      if (actor != p && actor instanceof Ghost) {
        if (p instanceof PacMan) {
          if (intersect(p, actor)) {
            collisionWithGhost(p, actor);
          }
        }
      }
    }
    return;
  }

  /**
   * handles the collision of a player with a ghost, sets the catched players position to the spawn
   */
  private void collisionWithGhost(Player p, Player actor) {
    if (gameState.superPower == 0) {
      gameState.scoreGhost += 50;
      p.setPosition(p.startPoint);
    } else {
      actor.position = actor.startPoint;
      gameState.scorePacMan += 50;
    }
  }

  /**
   * finds out the position of a ghost (has to be between walls in spawn) Diese Operation findet
   * eine Position für den Geist. Die Position muss zwischen der Wänden des Spawns sein.
   *
   * @return the random decided position of the ghost
   */
  private Point2D.Float findGhostPosition(List<Wall> spawn) {
    List<Point2D.Float> points = new ArrayList<>();
    for (Wall w : spawn) {
      for (int i = 0; i < w.points.size(); i++) {
        points.add(w.points.get(i));
      }
    }
    float maxX = 0;
    float maxY = 0;
    float minX = points.get(0).x;
    float minY = points.get(0).y;
    for (Point2D.Float point : points) {
      if (point.getX() > maxX) {
        maxX = (int) point.getX();
      }
      if (point.getY() > maxY) {
        maxY = (int) point.getY();
      }
      if (point.getX() < minX) {
        minX = (int) point.getX();
      }
      if (point.getY() < minY) {
        minY = (int) point.getY();
      }
    }
    Random rand = new Random();
    float x = rand.nextInt((int) (maxX - minX)) + minX;
    if (x == maxX) {
      x--;
    } else if (x == minX) {
      x++;
    }
    float y = rand.nextInt((int) (maxY - minY)) + minY;
    if (y == maxY) {
      y--;
    } else if (y == minY) {
      y++;
    }
    Point2D.Float position = new Point2D.Float(x, y);
    return position;
  }

  /**
   * displays the intersection of 2 player figures
   *
   * @param p player 1 of the intersection
   * @param actor player 2 of the intersection
   * @return true if intersect, else false
   */
  private boolean intersect(Player p, Player actor) {
    Point2D.Float displayActorPos = map.Pos2DisplayPos(p.position);
    Rectangle2D.Double pRect = new Rectangle2D.Double(displayActorPos.getX() - 3,
        displayActorPos.getY() - 3, map.tSize - 6, map.tSize - 6);
    Point2D displayPoint = map.Pos2DisplayPos(actor.position);
    Rectangle2D.Double actorRect = new Rectangle2D.Double(displayPoint.getX() - 3,
        displayPoint.getY() - 3, map.tSize - 6, map.tSize - 6);
    if (actorRect.intersects(pRect)) {
      return true;
    }
    return false;
  }

  /**
   * handles the collision by a pacman with a powerball, updates score and the game state, removes
   * the ball on the map
   *
   * @param o the powerball on the map
   */
  public void collisionWithPowerballPacMan(PowerBall o) {
    gameState.scorePacMan += 10;
    map.removePowerBall(o);
    gameState.superPower = 300;
    removeObject(o);
  }

  /**
   * handles the collision by a ghost with a powerball, updates score and the game state, removes
   * the ball on the map
   *
   * @param o the powerball on the map
   */
  public void collisionWithPowerballGhost(PowerBall o) {
    gameState.scoreGhost += 10;
    map.removePowerBall(o);
    gameState.superPower = 300;
    removeObject(o);
  }

  /**
   * removes the object from the map
   *
   * @param o the object getting removed
   */
  public void removeObject(Ball o) {
    for (Player actor : players) {
      actor.sendRemoveObject(o);
    }
  }

  /**
   * handles the collision by a pacman with a ball, updates score and the game state, removes the
   * ball on the map
   *
   * @param o the ball on the map
   */
  public void collisionWithBallPacMan(Ball o) {
    gameState.scorePacMan += 1;
    map.removeBall(o);
    removeObject(o);
  }

  /**
   * handles the collision by a ghost with a ball, updates score and the game state, removes the
   * ball on the map
   *
   * @param o the ball on the map
   */
  public void collisionWithBallGhost(Ball o) {
    gameState.scoreGhost += 1;
    map.removeBall(o);
    removeObject(o);
  }

  /**
   * "getter" for the PacMans
   *
   * @return players who are PacMans
   */
  public PacMan getPacman() {
    for (Player actor : players) {
      if (actor instanceof PacMan) {
        return (PacMan) actor;
      }
    }
    return null;
  }

  /**
   * checks if super power is active, while active the ghosts will get a message so they know the
   * PacMans are able to hunt them
   */
  public void checkSuperPower() {
    if (gameState.superPower > 0) {
      gameState.superPower -= 1;
      for (Player actor : players) {
        if (actor instanceof Ghost) {
          ((Ghost) actor).setFear(true);
        }
      }
    } else {
      for (Player actor : players) {
        if (actor instanceof Ghost) {
          ((Ghost) actor).setFear(false);
        }
      }
    }
  }

  /**
   * creates the balls on the map
   */
  private void createBalls() {
    float xStart = 0;
    float yStart = 0;
    for (Player actor : players) {
      if (actor instanceof PacMan) {
        xStart = actor.position.x;
        yStart = actor.position.y;
        break;
      }
    }
    findNextBall(xStart, yStart);
  }

  /**
   * checks if the fields are free on the right, left, bottom, top, if this is the case, this
   * operation generates another ball on the position recursive
   *
   * @param xStart start position on x axis
   * @param yStart start position on y axis
   */
  private void findNextBall(float xStart, float yStart) {
    if (map.isFree(xStart + 1, yStart)) {
      Point2D.Float point = new Point2D.Float(xStart + 1, yStart);
      createBall(point);
      findNextBall(xStart + 1, yStart);
    }
    if (map.isFree(xStart - 1, yStart)) {
      Point2D.Float point = new Point2D.Float(xStart - 1, yStart);
      createBall(point);
      findNextBall(xStart - 1, yStart);
    }
    if (map.isFree(xStart, yStart - 1)) {
      Point2D.Float point = new Point2D.Float(xStart, yStart - 1);
      createBall(point);
      findNextBall(xStart, yStart - 1);
    }
    if (map.isFree(xStart, yStart + 1)) {
      Point2D.Float point = new Point2D.Float(xStart, yStart + 1);
      createBall(point);
      findNextBall(xStart, yStart + 1);
    }
  }

  /**
   * creates new ball and adds it to the appropriate list
   *
   * @param point location of the new ball
   */
  public void createBall(Point2D.Float point) {
    Ball ball = new Ball(point);
    map.Balls.add(ball);
  }

  /**
   * decides which part of the game will happens next, depending on the game status
   */
  public synchronized void doPlay() {
    if (gameState.gameStatus == initialized) {
      doInitialized();
    }
    if (gameState.gameStatus == playing) {
      detectCollisions();
      nextMove();
    }
    if (gameState.gameStatus == sendingFinishInfo) {
      doSendFinishInfo();
    }
  }

  /**
   * sends the info after the game is finished
   */
  private void doSendFinishInfo() {

  }

  /**
   * method which does the initialisation for the player
   */
  private void doInitialized() {
    for (int i = 0; i < players.size(); i++) {
      Player p = players.get(i);
      p.setState(PlayerState.playing);

    }
  }

  /**
   * method which checks if the game is over/done
   *
   * @return true: game is over; false: game not over yet
   */
  public boolean isDone() {
    return gameState.gameStatus == GameStatus.done;
  }

  /**
   * "getter" for the game status
   *
   * @return the actual game status
   */
  public GameStatus getGameStatus() {
    return gameState.gameStatus;
  }

  /**
   * "setter" for the game status
   *
   * @param gameStatus the game status
   */
  public void setGameStatus(GameStatus gameStatus) {
    gameState.gameStatus = gameStatus;
  }

  @Override
  public Object getObjectAtPosition(Float position) {
    return map.getObject(position);
  }

  @Override
  public float getMapSize() {
    return map.getMapSize();
  }

  @Override
  public Float getPacmanPosition() {
    return getPacman().position;
  }

  @Override
  public Float getExitPosition() {
    return map.gate.position;
  }

  @Override
  public boolean isInSpawn(Ghost ghost) {
    List<Point2D.Float> points = new ArrayList<>();
    for (Wall w : map.Spawn) {
      for (int i = 0; i < w.points.size(); i++) {
        points.add(w.points.get(i));
      }
    }
    float maxX = 0;
    float maxY = 0;
    float minX = points.get(0).x;
    float minY = points.get(0).y;
    for (Point2D.Float point : points) {
      if (point.getX() > maxX) {
        maxX = (int) point.getX();
      }
      if (point.getY() > maxY) {
        maxY = (int) point.getY();
      }
      if (point.getX() < minX) {
        minX = (int) point.getX();
      }
      if (point.getY() < minY) {
        minY = (int) point.getY();
      }
    }
    if (ghost.position.getX() < maxX && ghost.position.getX() > minX && ghost.position.getY() < maxY
        && ghost.position.getY() > minY) {
      return true;
    }
    return false;
  }

  /**
   * "getter" for the game state
   *
   * @return the current game state
   */
  public GameState getGameState() {
    return gameState;
  }
}