package pacManReloaded.server;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Timer;
import java.util.TimerTask;
import java.util.stream.Collectors;
import pacManReloaded.common.LobbyState;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.messages.IMessage;
import pacManReloaded.common.model.Direction;
import pacManReloaded.common.model.Ghost;
import pacManReloaded.common.model.PacMan;

public class Lobby {

  private final int timerInterval = 1 * 100; // 0,1s
  private int gameInterval = 0;
  private LobbyState state;
  private String name;
  private ArrayList<Player> players = new ArrayList<>();
  private ArrayList<Thread> readyPlayer = new ArrayList<>();
  private HashMap<Player, Integer> winner = new HashMap<Player, Integer>();
  private Timer timer;
  private Server server;
  private GameLogic gameLogic;
  private int counter;
  private long diff;
  private long now;
  private long last = 0;
  private long avg;
  ArrayList<Long> messungen = new ArrayList<>();
  ArrayList<Long> averages = new ArrayList<>();
  private String messungenFileName = "messungen.txt";


  /**
   * Constructor of the Lobby
   *
   * @param name name of the lobby
   * @param server server the lobby is connected to
   */
  public Lobby(String name, Server server) {
    this.server = server;
    this.state = LobbyState.Open;
    this.name = name;
    timer = new Timer();
    timer.schedule(new TimerTask() {
      @Override
      public void run() {
        getServerTicks();
        if (state == LobbyState.Playing) {
          checkGameStatus(); // 2min.
        }
        checkLobbyStatus();
      }
    }, timerInterval, timerInterval);

  }

  /**
   * checks the game status by the timer and sets the new game status
   */
  private void checkGameStatus() {
    gameInterval += timerInterval;
    if (gameInterval > 120000) {  //2min. = 120000
      gameLogic.setGameStatus(GameStatus.done);
    }
  }

  /**
   * proofs if the bid of the player is an integer
   *
   * @param s - bid as a string
   * @return - true if bid is an integer, else (if not) false
   */
  private static boolean isInteger(String s) {
    if (s == null || s.isEmpty()) {
      return false;
    }
    for (int i = 0; i < s.length(); i++) {
      if (i == 0 && s.charAt(i) == '-') {
        if (s.length() == 1) {
          return false;
        } else {
          continue;
        }
      }
      if (!Character.isDigit(s.charAt(i))) {
        return false;
      }
    }
    return true;
  }

  /**
   * checks the status of the lobby (state and amount of players connected to the lobby)
   */
  private synchronized void checkLobbyStatus() {
    // if no players are connected to the server every lobby will be closed
    if (players.size() == 0) {
      state = LobbyState.Closed;
      gameLogic = null;
    } else if (state == LobbyState.Playing) {
      ArrayList<Player> toRemove = players.stream().filter(p -> p.getState() == PlayerState.done)
          .collect(Collectors.toCollection(ArrayList::new));
      toRemove.forEach(p -> {
        p.gameFinished(false);
        players.remove(p);
      });
      if (players.size() == 1) {
        Player p = players.get(0);
        players.remove(0);
        state = LobbyState.Closed;
        gameLogic = null;
        p.gameFinished(false);
      }
      if (gameLogic.getGameStatus()
          == GameStatus.done) {    // else if game is over, close lobby (set state to close),
        state = LobbyState.Closed;
        findWinner();
        gameLogic = null;
        players.forEach(p -> {
          p.gameFinished(isWinner(p));
          if (isWinner(p)) {
            server.playerWon(p.getClient().getMyName(), winner.get(p));
          }
        });
        players = new ArrayList<>();
      }
    } // state playing
    else if (state == LobbyState.Open && players.size() == 4 && readyPlayer.size() == 4) {
      state = LobbyState.Playing;
      gameLogic = new GameLogic(players, server);
    } else if (state == LobbyState.StartPlaying) {
      if (players.size() < 4) { // sorry, no enough player
        state = LobbyState.Open;
        players.get(0).sendBeginFailed();
      }
    }
    // if the round isnt over, the
    if (gameLogic != null) {
      gameLogic.doPlay();
    }
  }

  /**
   * decides which playing "site" is the winner of the game (PacMan or Ghost)
   */
  private void findWinner() {
    if (gameLogic.getGameState().scorePacMan <= gameLogic.getGameState().scoreGhost) {
      for (Player pa : players) {
        if (pa instanceof Ghost) {
          winner.put(pa, gameLogic.getGameState().scoreGhost);
        }
      }
    } else {
      for (Player pa : players) {
        if (pa instanceof PacMan) {
          winner.put(pa, gameLogic.getGameState().scorePacMan);
        }
      }
    }
  }

  /**
   * checks if the player is the winner of the game
   *
   * @param p the player checking
   * @return true - if the player is the winner; false - if not
   */
  private boolean isWinner(Player p) {
    if (winner.containsKey(p)) {
      return true;
    }
    return false;
  }

  /**
   * getter for the name of the lobby
   *
   * @return name of the lobby
   */
  public String getName() {
    return name;
  }

  /**
   * "getter" for the state of the lobby
   *
   * @return state of the lobby
   */
  public LobbyState getState() {
    return state;
  }

  /**
   * "setter" for the state of the lobby
   *
   * @param state lobby status
   */
  public void setState(LobbyState state) {
    this.state = state;
  }

  /**
   * method to check if the lobby contains player
   *
   * @param playerName nickname of the player
   * @return true if there are players in the lobby, else (if lobby is empty) false
   */
  public synchronized boolean hasPlayer(String playerName) {
    for (Player player : players) {
      if (player.getClient().getMyName().equals(playerName)) {
        return true;
      }
    }
    return false;
  }

  /**
   * method to add a player to the lobby
   *
   * @param thread thread of the server
   * @return true if player was added (if players in lobby are less than 4), else (if there are  or
   * more players in a lobby) false
   */
  public boolean addPlayer(ServerThread thread) {
    if (players.size() >= 4) {
      return false;
    }
    Player newPlayer = new Player(thread);
    players.add(newPlayer);
    return true;
  }

  /**
   * method to remove a player from the server
   *
   * @param thread thread of the server
   */
  public void removePlayer(ServerThread thread) {
    Player p = null;
    for (Player player : players) {
      if (player.getClient() == thread) {
        p = player;
        break;
      }
    }
    if (p != null) {
      players.remove(p);
    }
  }

  /**
   * handles messages of the game
   *
   * @param gameMsg message of the game
   * @param thread thread of the server
   */
  public synchronized void handleMessage(IMessage gameMsg, ServerThread thread) {
    switch (gameMsg.getType()) {
      case BEGIN_GAME: {
        if (this.state == LobbyState.Open) {
          if (!readyPlayer.contains(thread)) {
            readyPlayer.add(thread);
          }
        }
        break;
      }
      case SEND_TO_ADDRESS: {
        handleBroadcastLobby((ChatWithAddress) gameMsg, thread);
        break;
      }
    }
  }

  /**
   * handles the broadcasting of the message in the client, sends the message to every connected
   * player
   *
   * @param msg message which gets broadcasted
   * @param thread thread of the server
   */
  private void handleBroadcastLobby(ChatWithAddress msg, ServerThread thread) {
    System.out.println("Lobby handle broadcast " + thread.getMyName());
    msg.setAddress(thread.getMyName()); // replace "to" with "from"

    players.stream().forEach(p -> {
      if (p.getClient() != thread) { // don't send it back
        p.getClient().send(msg);
      }
    });
  }

  /**
   * handles the changing of each players moving direction
   *
   * @param changeDirMsg message with the informations about the direction
   * @param thread the thread of the server
   */
  public void handleChangeDir(String changeDirMsg, ServerThread thread) {
    for (Player p : players) {
      if (p.getClient() == thread) {
        p.direction = Direction.valueOf(changeDirMsg);
      }
    }
  }

  /**
   * "getter" of the players
   *
   * @return all players
   */
  public ArrayList<Player> getPlayers() {
    return players;
  }

  /**
   * messure of the server ticks in milli seconds, counts them to be able to analyse them
   */
  public void getServerTicks() {
    if (counter == 0) {
      last = System.currentTimeMillis();
    }
    if (counter > 0 && counter < 101) {
      now = System.currentTimeMillis();
      diff = now - last;
      last = now;
      messungen.add(diff);
      System.out.println("Messung:" + diff);
    }
    if (counter == 101) {
      try {
        PrintWriter pw = new PrintWriter(new FileOutputStream(messungenFileName));
        long sum = 0;
        for (Long d : messungen) {
          sum += d;
        }
        avg = sum / messungen.size();
        System.out.println("Average:" + avg);
        averages.add(avg);
        for (Long a : averages) {
          pw.println(a.toString());
        }
        pw.close();
      } catch (FileNotFoundException e) {
        System.out.println("Cannot write Server-Ticks: " + e.getMessage());
      }
    }
    counter++;
  }
}