package pacManReloaded.server;

import static pacManReloaded.common.Command.ACTOR_POSITION;
import static pacManReloaded.common.Command.BEGIN_FAILED;
import static pacManReloaded.common.Command.GAMESTATE;
import static pacManReloaded.common.Command.GAME_FINISHED;
import static pacManReloaded.common.Command.INIT_GAMESTATE;
import static pacManReloaded.common.Command.INIT_PLAYER;
import static pacManReloaded.common.Command.MAP;
import static pacManReloaded.common.Command.REMOVE_OBJECT;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import pacManReloaded.common.CommandFactory;
import pacManReloaded.common.messages.IMessage;
import pacManReloaded.common.model.Ball;
import pacManReloaded.common.model.Direction;
import pacManReloaded.common.model.Gate;
import pacManReloaded.common.model.Ghost;
import pacManReloaded.common.model.INavigator;
import pacManReloaded.common.model.Map;
import pacManReloaded.common.model.PacMan;
import pacManReloaded.common.model.PowerBall;
import pacManReloaded.common.model.Wall;

public class Player {

  public Direction direction;
  public Point2D.Float position;
  protected float speed;


  protected INavigator navigator;
  public Point2D.Float startPoint;
  private ServerThread client;
  private PlayerState state;
  private long msgSentTime; // used later to check protocol's timeout
  private int points;

  /**
   * Constructor of the Player
   *
   * @param client thread of the client
   */
  public Player(ServerThread client) {
    this.client = client;
  }

  /**
   * dummy-constructor
   */
  public Player() {

  }

  /**
   * "getter" for the client(thread)
   *
   * @return the (thread of the) client
   */
  public ServerThread getClient() {
    return client;
  }

  /**
   * "getter" for the state of the player
   *
   * @return state of the player
   */
  public PlayerState getState() {
    return state;
  }

  /**
   * creates the map and sends it, sets postions of each object
   *
   * @param map the map
   */
  public void sendMap(Map map) {
    ArrayList<String> params = new ArrayList();
    String object = map.xSize + "," + map.ySize + "," + map.tSize + ";";
    params.add(object);
    object = "";
    for (Wall w : map.Walls) {
      for (Point2D p : w.points) {
        object = object + p.getX() + "," + p.getY() + ";";
      }
      object = object + "!";
    }
    params.add(object);
    object = "";
    for (Wall s : map.Spawn) {
      for (Point2D p : s.points) {
        object = object + p.getX() + "," + p.getY() + ";";
      }
      object = object + "!";
    }
    params.add(object);
    object = "";
    for (PowerBall b : map.PowerBalls) {
      object = object + b.position.getX() + "," + +b.position.getY() + ";";
    }
    params.add(object);
    object = "";
    for (Ball b : map.Balls) {
      object = object + b.position.getX() + "," + +b.position.getY() + ";";
    }
    params.add(object);
    object = map.gate.position.x + "," + map.gate.position.y + ";";
    params.add(object);
    IMessage mapMsg = CommandFactory.createMessage(MAP, params);
    client.send(mapMsg);
  }

  /**
   * initialization of the players, decides type of the actor, if PacMan or Ghost
   *
   * @param players list of all players
   */
  public void sendInitPlayer(ArrayList<Player> players) {
    ArrayList<String> params = new ArrayList();
    String parm = "";
    for (Player p : players) {
      parm = parm + p.client.getMyName();
      if (p instanceof PacMan) {
        parm = parm + ",PacMan," + p.position.getX() + "," + p.position.getY() + ";";
      } else {
        parm =
            parm + ",Ghost," + ((Ghost) p).color + "," + p.position.getX() + "," + p.position.getY()
                + ";";
      }
    }
    params.add(parm);
    IMessage initplayerMsg = CommandFactory.createMessage(INIT_PLAYER, params);
    client.send(initplayerMsg);
  }

  /**
   * initialization of the gamestate, sets the status and send it
   *
   * @param gameState current game state
   */
  public void sendInitGameState(GameState gameState) {
    ArrayList<String> params = new ArrayList();
    params.add(parmsGameState(gameState));
    IMessage gameStateMsg = CommandFactory.createMessage(INIT_GAMESTATE, params);
    client.send(gameStateMsg);
  }

  /**
   * updates the removal of an object, when a PacMan collects a ball
   *
   * @param o ball to remove
   */
  public void sendRemoveObject(Ball o) {
    ArrayList<String> params = new ArrayList();
    String parm = o.position.getX() + "," + o.position.getY();
    params.add(parm);
    IMessage gameStateMsg = CommandFactory.createMessage(REMOVE_OBJECT, params);
    client.send(gameStateMsg);
  }

  /**
   * updates the game state and sends it
   *
   * @param gameState the game state
   */
  public void sendGameState(GameState gameState) {
    ArrayList<String> params = new ArrayList();
    params.add(parmsGameState(gameState));
    IMessage gameStateMsg = CommandFactory.createMessage(GAMESTATE, params);
    client.send(gameStateMsg);
  }

  /**
   * creates a string for the game state
   *
   * @param gameState the game state
   */
  private String parmsGameState(GameState gameState) {
    String parm = "";
    parm = gameState.superPower + "," + gameState.scoreGhost + "," + gameState.scorePacMan;
    return parm;
  }

  /**
   * sends the position of the player and updates it
   */
  public void sendActorPosition(ArrayList<Player> players) {
    ArrayList<String> params = new ArrayList();
    String parm = "";
    for (Player p : players) {
      parm = parm + p.client.getMyName();
      parm = parm + "," + p.direction + "," + p.position.getX() + "," + p.position.getY() + ";";
    }
    params.add(parm);
    IMessage actorPosMsg = CommandFactory.createMessage(ACTOR_POSITION, params);
    client.send(actorPosMsg);
  }

  /**
   * "setter" for the state of the player
   *
   * @param state state of the player
   */
  public void setState(PlayerState state) {
    this.state = state;
  }

  /**
   * "getter" for the message send time
   *
   * @return the sending time of the message
   */
  public long getMsgSentTime() {
    return msgSentTime;
  }

  /**
   * sends the message for the failed begin of a game to the player
   */
  public void sendBeginFailed() {
    IMessage beginFailedMsg = CommandFactory.createMessage(BEGIN_FAILED, null);
    client.send(beginFailedMsg);
  }

  /**
   * method which handles the finished game
   *
   * @param gameResult - gives the result as boolean. true equals a win and false equals a loss.
   */
  public void gameFinished(boolean gameResult) {
    ArrayList<String> params = new ArrayList<>();
    if (gameResult) {
      params.add("WON");
    } else {
      params.add("LOSE");
    }
    params.add("");
    IMessage msg = CommandFactory.createMessage(GAME_FINISHED, params);
    client.send(msg);
    msgSentTime = System.currentTimeMillis();
  }

  /**
   * checks the direction of the player, if possible the position of the player will be updated
   */
  protected void changePosition() {
    Object o;
    switch (direction) {
      case right:
        o = navigator.getObjectAtPosition(new Point2D.Float(position.x + speed, position.y));
        if (!(o instanceof Wall)) {
          position = new Point2D.Float((position.x + speed), position.y);
          if (position.x > navigator.getMapSize() - 1) {
            position.x = 0;
          } else if (position.x < 0) {
            position.x = navigator.getMapSize() - 1;
          }
        }
        break;
      case left:
        o = navigator.getObjectAtPosition(new Point2D.Float(position.x - speed, position.y));
        if (!(o instanceof Wall) && !(o instanceof Gate)) {
          position = new Point2D.Float((position.x - speed), position.y);
          if (position.x > navigator.getMapSize() - 1) {
            position.x = 0;
          } else if (position.x < 0) {
            position.x = navigator.getMapSize() - 1;
          }
        }
        break;
      case up:
        o = navigator.getObjectAtPosition(new Point2D.Float(position.x, position.y - speed));
        if (!(o instanceof Wall) && !(o instanceof Gate)) {
          position = new Point2D.Float((position.x), (position.y - speed));
          if (position.x > navigator.getMapSize() - 1) {
            position.x = 0;
          } else if (position.x < 0) {
            position.x = navigator.getMapSize() - 1;
          }
        }
        break;
      case down:
        o = navigator.getObjectAtPosition(new Point2D.Float(position.x, position.y + speed));
        if (!(o instanceof Wall) && !(o instanceof Gate)) {
          position = new Point2D.Float((position.x), (position.y + speed));
          if (position.x > navigator.getMapSize() - 1) {
            position.x = 0;
          } else if (position.x < 0) {
            position.x = navigator.getMapSize() - 1;
          }
        }
        break;
      default:
        break;
    }
  }

  public void nextMove() {
  }

  /**
   * "setter" for the starting point of a player
   *
   * @param point the position of the map (startpoint)
   */
  public void setStartPoint(Point2D.Float point) {
    startPoint = point;
  }

  /**
   * "setter" for the position of an object
   *
   * @param point the position of the map
   */
  public void setPosition(Point2D.Float point) {
    position = point;
  }

  /**
   * "getter" for the direction of the player
   *
   * @return the direction
   */
  public Direction getDirection() {
    return direction;
  }

  /**
   * "setter" for the direction of a player
   *
   * @param direction players direction
   */
  public void setDirection(Direction direction) {
    this.direction = direction;
  }

}
