package pacManReloaded.server;

import static pacManReloaded.common.Command.CLOSE;
import static pacManReloaded.common.Command.PING;

import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import pacManReloaded.common.CommandFactory;
import pacManReloaded.common.JoinLobbyResult;
import pacManReloaded.common.LobbyState;
import pacManReloaded.common.messages.ChatMessage;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.messages.IMessage;

public class Server implements Runnable {

  private ArrayList<ServerThread> clients = new ArrayList();
  private ArrayList<Lobby> lobbies = new ArrayList<>();
  private ServerSocket server;
  private Thread thread = null;
  private Timer timer;
  private final int pingInterval = 5 * 1000; //50s
  private String highScoreFileName = "hs.txt";
  private ArrayList<HighScore> highScores;
  private String manualFileName = "manual.txt";
  private String manualText = "";


  /**
   * Constructor of the server, opens a new socket
   *
   * @param port - port of the server
   */
  public Server(int port) {
    InetAddress iP = null;
    try {
      iP = InetAddress.getLocalHost();
    } catch (UnknownHostException e) {
      e.printStackTrace();
    }

    Path currentRelativePath = Paths.get("");
    String s = currentRelativePath.toAbsolutePath().toString();
    System.out.println("Current relative path is: " + s);
    System.out.println("Your IP-Address is: " + iP.getHostAddress());

    try {
      System.out.println("Binding to port " + port + ", please wait  ...");
      server = new ServerSocket(port);
      loadHighScores();
      loadManual();
      System.out.println("Server started: " + server);
    } catch (IOException ioe) {
      System.err.println(ioe.toString());
      System.exit(1);
    }
    start();
  }

  /**
   * starts the thread of the server
   */
  private void start() {
    if (thread == null) {
      thread = new Thread(this);
      timer = new Timer();
      timer.schedule(new TimerTask() {
        @Override
        public void run() {
          if (clients.size() > 0) {
            checkPingPong();
          }
        }
      }, pingInterval, pingInterval);
      thread.start();
    }
  }

  /**
   * when the run method starts, the server is accepting connection at it's socket, then starts a
   * ClientThread, which handles the Client and afterwards accepts new connections
   */
  public void run() {
    try {
      while (thread != null) {
        System.out.println("Waiting for a client");
        addThread(server.accept());
      }
      try {
        server.close();
        for (ServerThread serverThread : clients) {
          try {
            serverThread.close();
          } catch (IOException e283) {
          }
        }
      } catch (IOException e212) {
        System.out.println("Error occured during closing the Server! (IOException)");
      }
    } catch (IOException e1) {
      System.out.println("connection failed");
    }
  }

  /**
   * method which handles the removal of a client,
   *
   * @param threadToTerminate - thread which will be removed/terminated
   */
  public synchronized void remove(ServerThread threadToTerminate) {
    if (!clients.contains(threadToTerminate)) {
      return;
    }
    System.out.println("Removing client thread " + threadToTerminate.getThreadId());
    if (threadToTerminate.getSocket() != null) {
      IMessage closeMsg = CommandFactory.createMessage(CLOSE, null);
      threadToTerminate.send(closeMsg);
    }
    clients.remove(threadToTerminate);
    for (Lobby lobby : lobbies) {
      lobby.removePlayer(threadToTerminate);
    }
    System.out.println("Client thread " + threadToTerminate.getThreadId() + " removed");
    try {
      threadToTerminate.close();
    } catch (IOException ioe) {
      System.out.println("Error closing thread: " + ioe);
    }
  }

  /**
   * checks if clients ar still connected with the server by sending a PING-message to all clients.
   * If PING is not returned after 4 intervalls, the client disconnects from the server.
   */
  private synchronized void checkPingPong() {
    ArrayList<ServerThread> toRemove = new ArrayList<>();
    long now = System.currentTimeMillis();
    System.out.println("Ping-pong check..." + toReadableTime(now));
    IMessage pingMsg = CommandFactory.createMessage(PING, null);
    for (ServerThread client : clients) {
      if (now - client.getLastContact() > 4 * pingInterval) {
        toRemove.add(client);
        System.out.println(
            "Client to remove: " + client.getMyName() + "last time: " + toReadableTime(
                client.getLastContact())
        );
      } else if (now - client.getLastContact() > pingInterval) {
        client.send(pingMsg);
        System.out
            .println("Ping sent to client " + client.getMyName() + " at " + toReadableTime(now));
      }
    }
    for (ServerThread client : toRemove) {
      remove(client);
    }
  }

  /**
   * converts the given time into a readable String.
   *
   * @param timeInms - given time in milliseconds.
   * @return - converted time as a String
   */
  private String toReadableTime(long timeInms) {
    SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
    Date resultDate = new Date(timeInms);
    return sdf.format(resultDate);
  }

  /**
   * adds the thread/client to the server
   *
   * @param socket - socket of the server
   */
  private void addThread(Socket socket) {
    ServerThread st = new ServerThread(this, socket);
    clients.add(st);
    System.out.println("Client accepted: " + socket + ". Connected " + clients.size() + " clients");
    try {
      st.open();
      st.start();
    } catch (IOException ioe) {
      System.out.println("Error opening thread: " + ioe);
      clients.remove(st);
    }

  }

  /**
   * Gets the Nickname that should be used. Checks the uniqueness of it and sets the unique Nickname
   * of the Client.
   *
   * @param thread - Thread of the Server.
   * @param myName - The Nickname that should be set.
   * @param suffix - The suffix for the Nickname.
   * @return the set Nickname of the Client.
   */
  public synchronized String getAndSetName(ServerThread thread, String myName, String suffix) {
    String newName = myName + suffix;
    // replace separator...
    newName = newName.replace(IMessage.separator, "");
    if (newName.length() == 0) {
      newName = "ABC";
    }

    final String checkName = newName;
    if (clients.size() > 1) {
      if (clients.stream()
          .anyMatch(c -> c.getMyName() != null && c.getMyName().equals(checkName))) {
        String newSuffix =
            suffix.length() == 0 ? "2" : String.valueOf(Integer.parseInt(suffix) + 1);
        newName = getAndSetName(thread, myName, newSuffix);
        thread.setMyName(newName);
      }
    }
    if (suffix.length() == 0) {
      thread.setMyName(newName);
    }
    return newName;

  }

  /**
   * method which makes the highscore of the file to a readable string
   *
   * @return the highscore as a readable string
   */
  public String getHighScoreString() {
    String result = "High Score:" + IMessage.separator + IMessage.separator;
    for (HighScore h : highScores) {
      result +=
          "Client Name: " + h.getName() + "; \t Number: " + h.getNumber() + "; \t HighScore: " + h
              .getHighScore() + ";" + IMessage.separator;
    }
    return result;
  }

  /**
   * "getter" for the info of the lobby
   *
   * @return - info of the lobby
   */
  public String getLobbyInfoString() {
    String result = "Lobbies: " + IMessage.separator;
    for (Lobby l : lobbies) {
      result +=
          "  Lobby Name: " + l.getName() + ";  Status: " + l.getState() + ";" + IMessage.separator;
      result += "    Users:" + IMessage.separator;
      for (Player p : l.getPlayers()) {
        result +=
            "      Username: " + p.getClient().getMyName() + ";  State: " + p.getState() + ";"
                + IMessage.separator;
      }
      result += IMessage.separator;
    }
    return result;
  }

  /**
   * get the names of the connected clients in a string
   *
   * @return the connected users
   */
  public String getClientInfoString() {
    String result = "Connected users: " + IMessage.separator;
    for (ServerThread st : clients) {
      result +=
          "   " + st.getMyName()
              + IMessage.separator;
    }
    return result;
  }

  /**
   * loads the manual from a file
   *
   * @throws IOException exception if file not existing
   */
  private void loadManual() throws IOException {
    InputStream is = this.getClass().getClassLoader()
        .getResourceAsStream(manualFileName);
    BufferedReader in = new BufferedReader(new InputStreamReader(is));
    String line;
    while ((line = in.readLine()) != null) {
      manualText += line + System.lineSeparator();
    }
    in.close();
  }

  /**
   * loads the highscore from a saved file
   */
  private void loadHighScores() {
    highScores = new ArrayList<>();
    InputStream is = this.getClass().getClassLoader()
        .getResourceAsStream(highScoreFileName);
    try {
      BufferedReader in = new BufferedReader(
          new InputStreamReader(is));
      String line = in.readLine();
      while (line != null) {
        String[] splitted = line.split("\\|");
        HighScore hs = new HighScore(splitted[0], Integer.parseInt(splitted[1]),
            Integer.parseInt(splitted[2]));
        highScores.add(hs);
        line = in.readLine();
      }
      in.close();
    } catch (Exception e) {
      System.out.println("Cannot load high scores: " + e.getMessage());
    }
  }

  /**
   * saves the highscore in a file
   */
  private void saveHighScores() {
    try {
      PrintWriter pw = new PrintWriter(new FileOutputStream(highScoreFileName));
      for (HighScore hs : highScores) {
        pw.println(hs.toString());
      }
      pw.close();
    } catch (Exception e) {
      System.out.println("Cannot save high scores: " + e.getMessage());
    }
  }

  /**
   * "getter" for the manual text as string
   *
   * @return the manual as a string
   */
  public String getManualText() {
    String result = manualText.replace(System.lineSeparator(), IMessage.separator);
    return result;
  }

  /**
   * handles the broadcast of a message in the server, sends the message to all clients connected to
   * the server
   *
   * @param thread - thread of the server
   * @param chatMessage - message which has to be broadcasted
   */
  public void broadcast(ServerThread thread, ChatMessage chatMessage) {
    chatMessage.setFrom(thread.getMyName());
    System.out.println("Broadcasting info from " + thread.getMyName());
    for (ServerThread client : clients) {
      if (client != thread) {
        client.send(chatMessage);
        System.out.println("Chat sent to " + client.getMyName());
      }
    }
  }

  /**
   * handles private messages
   *
   * "NotInLobby" is reserved as keyword
   *
   * @param thread - thread of the server
   * @param msg - the private message
   */
  public synchronized void handleChatWithAddress(ServerThread thread, ChatWithAddress msg) {
    System.out.println("Forward chat message from " + thread.getMyName());
    String address = msg.getAddress();
    if (address.equals("NotInLobby")) {
      msg.setAddress(thread.getMyName());
      ArrayList<ServerThread> clientsIsInLobby = new ArrayList();
      for (ServerThread c : clients) {
        for (Lobby l : lobbies) {
          if (l.hasPlayer(c.getMyName())) {
            clientsIsInLobby.add(c);
          }
        }
        if (!clientsIsInLobby.contains(c) && c != thread) {
          c.send(msg);
        }
      }

    } else {
      Optional<Lobby> optionalLobby = lobbies.stream()
          .filter(l -> l.getName().equals(address) && l.hasPlayer(thread.getMyName()))
          .findFirst();
      if (optionalLobby.isPresent()) {
        Lobby lo = optionalLobby.get();
        lo.handleMessage(msg, thread);
        System.out.println("Chat broadcasted in lobby" + thread.getMyName());
        return;

      }
    }

    Optional<ServerThread> optionalCl = clients.stream().filter(c -> c.getMyName().equals(address))
        .findFirst();
    if (optionalCl.isPresent()) {
      ServerThread cl = optionalCl.get();
      msg.setAddress(thread.getMyName()); // replace "to" with "from"
      cl.send(msg);
      System.out.println("Whisper chat to " + cl.getMyName());
      return;
    }
    System.out.println("Client " + address + " not found!");
  }

  /**
   * Getter for the function toReadableTime.
   *
   * @param timeInms - the time that should be brought in a Readable time format in Milli-seconds.
   * @return the Return value of the function toReadableTime in the wished time format.
   */
  public String getToReadableTime(long timeInms) {
    return toReadableTime(timeInms);
  }

  /**
   * method which handles the accession into a lobby
   *
   * @param thread - thread of the server
   * @param lobbyName - name of the lobby
   * @return - state of the accession result
   */
  public synchronized JoinLobbyResult joinLobby(ServerThread thread, String lobbyName) {
    lobbyName = lobbyName.replace('|', '_');
    Lobby shouldJoin = null;
    for (Lobby lobby : lobbies) {
      if (lobby.hasPlayer(thread.getMyName())) {
        return JoinLobbyResult.AlreadyPlaying;
      }
      if (lobby.getName().equalsIgnoreCase(lobbyName)) {
        if (lobby.getState() != LobbyState.Open) {
          return JoinLobbyResult.Rejected;
        }
        shouldJoin = lobby;
      }
    }
    if (shouldJoin != null) {
      if (shouldJoin.addPlayer(thread)) {
        return JoinLobbyResult.Joined;
      }
      return JoinLobbyResult.Rejected;
    }

    Lobby newLobby = new Lobby(lobbyName, this);
    if (!newLobby.addPlayer(thread)) {
      return JoinLobbyResult.Rejected;
    }
    lobbies.add(newLobby);
    return JoinLobbyResult.Created;
  }

  /**
   * method to find a lobby
   *
   * @param thread - thread of the server
   * @return - the lobby
   */
  public Lobby findLobby(ServerThread thread) {
    for (Lobby lobby : lobbies) {
      if (lobby.hasPlayer(thread.getMyName())) {
        return lobby;
      }
    }
    return null;
  }

  /**
   * includes the player who won into the highscore txt file
   *
   * @param playerName the name of the plasyer who won the game
   */
  public void playerWon(String playerName, int score) {
    for (HighScore highScore : highScores) {
      if (highScore.getName().equalsIgnoreCase(playerName)) {
        highScore.setNumber(highScore.getNumber() + 1);
        if (highScore.getHighScore() < score) {
          highScore.setHighScore(score);
        }
        saveHighScores();
        return;
      }
    }
    highScores.add(new HighScore(playerName, 1, score));
    saveHighScores();
  }
}