package pacManReloaded.server;

import static pacManReloaded.common.Command.CLIENT_INFO;
import static pacManReloaded.common.Command.GET_NICKNAME;
import static pacManReloaded.common.Command.HIGHSCORE;
import static pacManReloaded.common.Command.JOIN_LOBBY_RESPONSE;
import static pacManReloaded.common.Command.LOBBY_INFO;
import static pacManReloaded.common.Command.MANUAL;
import static pacManReloaded.common.Command.PONG;
import static pacManReloaded.common.Command.SET_NICKNAME;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import pacManReloaded.common.CommandFactory;
import pacManReloaded.common.JoinLobbyResult;
import pacManReloaded.common.messages.ChatMessage;
import pacManReloaded.common.messages.ChatWithAddress;
import pacManReloaded.common.messages.IMessage;
import pacManReloaded.common.messages.NoParameterMessage;
import pacManReloaded.common.messages.SingleStringParameterMessage;

public class ServerProtocol {

  /**
   * on init method
   *
   * @param st the server thread
   */
  public void onInit(ServerThread st) {
    st.send(CommandFactory.createMessage(GET_NICKNAME, null));
  }

  /**
   * makes the time to a readable string
   *
   * @param timeInms time in milliseconds
   * @return time as a string
   */
  private String toReadableTime(long timeInms) {
    SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
    Date resultDate = new Date(timeInms);
    return sdf.format(resultDate);
  }

  /**
   * method which handles the messages
   *
   * @param thread the thread of the server
   * @param gameMsg the message which has to get handled
   * @throws IOException exception for wrong params
   */
  public void handleMsg(ServerThread thread, IMessage gameMsg) throws IOException {
    System.out.println("Server " + thread.getThreadId() + " received: " + gameMsg);
    switch (gameMsg.getType()) {
      /**
       * Nickname of the Client will be checked if unique and send back.
       */
      case MY_NICKNAME:
        SingleStringParameterMessage myNameMessage = (SingleStringParameterMessage) gameMsg;
        String acceptedName = thread.getServer()
            .getAndSetName(thread, myNameMessage.getParamValue(), "");
        ArrayList<String> params = new ArrayList();
        params.add(acceptedName);
        IMessage response = CommandFactory.createMessage(SET_NICKNAME, params);
        thread.send(response);
        break;
      /**
       *  gets a Message from the Client and broadcasts it to all Clients.
       */
      case MESSAGE:
        ChatMessage chatMessage = (ChatMessage) gameMsg;
        thread.getServer().broadcast(thread, chatMessage);
        break;
      case SEND_TO_ADDRESS:
        ChatWithAddress chatMsg = (ChatWithAddress) gameMsg;
        thread.getServer().handleChatWithAddress(thread, chatMsg);
        break;
      case JOIN_LOBBY:
        SingleStringParameterMessage joinLobbyMsg = (SingleStringParameterMessage) gameMsg;
        JoinLobbyResult lobby = thread.getServer().joinLobby(thread, joinLobbyMsg.getParamValue());
        params = new ArrayList();
        params.add(lobby.toString());
        response = CommandFactory.createMessage(JOIN_LOBBY_RESPONSE, params);
        thread.send(response);
        break;
      case CHANGE_DIRECTION:
        SingleStringParameterMessage changeDirMsg = (SingleStringParameterMessage) gameMsg;
        Lobby lo = thread.getServer().findLobby(thread);
        if (lo != null) {
          lo.handleChangeDir(changeDirMsg.getParamValue(), thread);
        } else {
          System.out.println("No lobby found for " + thread.getMyName());
        }
        break;
      case GET_LOBBY_INFO: {
        String serverInfo = thread.getServer().getLobbyInfoString();
        params = new ArrayList<>();
        params.add(serverInfo);
        response = CommandFactory.createMessage(LOBBY_INFO, params);
        thread.send(response);
        break;
      }
      case GET_CLIENT_INFO: {
        String serverInfo = thread.getServer().getClientInfoString();
        params = new ArrayList<>();
        params.add(serverInfo);
        response = CommandFactory.createMessage(CLIENT_INFO, params);
        thread.send(response);
        break;
      }
      case GET_HIGHSCORE: {
        String serverInfo = thread.getServer().getHighScoreString();
        params = new ArrayList<>();
        params.add(serverInfo);
        response = CommandFactory.createMessage(HIGHSCORE, params);
        thread.send(response);
        break;
      }
      case GET_MANUAL: {
        String serverInfo = thread.getServer().getManualText();
        params = new ArrayList<>();
        params.add(serverInfo);
        response = CommandFactory.createMessage(MANUAL, params);
        thread.send(response);
        break;
      }
      case BEGIN_GAME: {
        Lobby l = thread.getServer().findLobby(thread);
        if (l != null) {
          l.handleMessage(gameMsg, thread);
        } else {
          System.out.println("No lobby found for " + thread.getMyName());
        }
        break;
      }
      /**
       * Sends a Ping to the Client to check the connection.
       */
      case PING:
        IMessage pongMsg = new NoParameterMessage(PONG);
        thread.send(pongMsg);
        System.out.println("Pong sent as answer at: " + toReadableTime(System.currentTimeMillis()));
        break;
      /**
       * Sends a PONG to the Client to finish the connection test successfully.
       */
      case PONG:
        thread.updateLastContact();
        System.out.println(
            "Server received a Pong from " + thread.getMyName() + " at " + toReadableTime(
                System.currentTimeMillis())
        );
        break;
      /**
       * Is called when a Unknown Message type occurs.
       * Informs via the Protocol output that a undefined command was used.
       */
      case UNKNOWN_MESSAGE:
        System.out
            .println("This command is not defined: " + gameMsg + " and therefore will be ignored.");
        break;
      default:
        System.out.println("We should not be here! " + gameMsg);
        break;
    }
  }
}