package pacManReloaded.server;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import pacManReloaded.common.CommandFactory;
import pacManReloaded.common.messages.IMessage;

public class ServerThread extends Thread {

  private Socket socket;
  private int id;
  private BufferedReader streamIn = null;
  private BufferedWriter streamOut = null;
  private ServerProtocol protocol = new ServerProtocol();

  private Server server;
  private String myName;
  private long lastContact;

  /**
   * Constructor of the server thread
   *
   * @param serverx the server
   * @param socketx the socket of the server
   */
  public ServerThread(Server serverx, Socket socketx) {
    super();
    server = serverx;
    socket = socketx;
    id = socket.getPort();
    lastContact = System.currentTimeMillis();
  }

  /**
   * run method listens to the input stream, processes the data that it receives
   */
  public void run() {
    System.out.println("Server Thread " + id + " waiting for messages...");
    protocol.onInit(this);
    while (true) {
      try {
        String msg = streamIn.readLine();
        if (msg != null) {
          IMessage gameMsg = CommandFactory.parseMessage(msg);
          protocol.handleMsg(this, gameMsg);
        } else {
          break;
        }
      } catch (IOException ioe) {
        System.out.println(id + " : " + ioe.getMessage());
        break;
      }
    }
    try {
      close();
    } catch (IOException e) {
      e.printStackTrace();
    }
    System.out.println(id + " : removed!");
    server.remove(this);
  }

  /**
   * opens the Input- and OutputStream for the socket
   *
   * @throws IOException exception for wrong input
   */
  public void open() throws IOException {
    streamIn = new BufferedReader(new InputStreamReader(socket.getInputStream(), "utf-8"));
    streamOut = new BufferedWriter(new OutputStreamWriter(socket.getOutputStream(), "utf-8"));
  }

  /**
   * closes the socket of the server
   *
   * @throws IOException exception if socket isnt closed correctly
   */
  public void close() throws IOException {
    if (socket != null) {
      socket.close();
      socket = null;
    }
    if (streamIn != null) {
      streamIn.close();
      streamIn = null;
    }
    if (streamOut != null) {
      streamOut.close();
      streamOut = null;
    }
    server.remove(this);
  }

  /**
   * method of the sending of a message
   *
   * @param msg - the message
   */
  public void send(IMessage msg) {
    try {
      streamOut.write(msg + System.lineSeparator());
      streamOut.flush();
    } catch (IOException ioe) {
      System.out.println(id + " ERROR sending: " + ioe.getMessage());
      server.remove(this);
    }
  }

  /**
   * method to make time to a readable string
   *
   * @param timeInms - time in milliseconds
   * @return - the time as a string
   */
  private String toReadableTime(long timeInms) {
    SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
    Date resultDate = new Date(timeInms);
    return sdf.format(resultDate);
  }

  /**
   * "getter" for the server
   *
   * @return - the server
   */
  public Server getServer() {
    return server;
  }

  /**
   * "getter" for the id of the thread
   *
   * @return - the id of the thread
   */
  public int getThreadId() {
    return id;
  }

  /**
   * "getter" for the nickname of the player
   *
   * @return - nickname of the player
   */
  public String getMyName() {
    return this.myName;
  }

  /**
   * "setter" for the nickname of the player
   *
   * @param myName - nickname of the player
   */
  public void setMyName(String myName) {
    this.myName = myName;
  }

  /**
   * "getter" for the last contact of the server
   *
   * @return - last contact of the server (the time)
   */
  public long getLastContact() {
    return lastContact;
  }

  /**
   * updates the last contact of the server in milliseconds
   */
  public void updateLastContact() {
    this.lastContact = System.currentTimeMillis();
  }

  /**
   * "getter" for the socket
   *
   * @return - the socket
   */
  public Socket getSocket() {
    return socket;
  }


}
// TODO fixing problem with wrong writing out of the ServerTicks
